package main

import (
	"bufio"
	"embed"
	"fmt"
	"os"
	"strconv"
	"strings"

	"code.justin.tv/cplat/twitchling/localization/bundle"
	"code.justin.tv/cplat/twitchling/localization/language"
	"code.justin.tv/cplat/twitchling/localization/message"
)

const (
	Reset = "\u001b[0m"
	Bold  = "\u001b[1m"

	raidMessage = bundle.MessageID("raid")
)

//go:embed localization/translations/*.json
var translationBundle embed.FS

type Raid struct {
	User    string `twitchling:"user,userID"`
	Viewers int    `twitchling:"viewers"`
}

func main() {
	b, _ := bundle.New(translationBundle)
	reader := bufio.NewReader(os.Stdin)

	for {
		fmt.Print("Enter language: ")
		langInput, _ := reader.ReadString('\n')
		langInput = strings.ReplaceAll(langInput, "\n", "")
		lang, _ := language.NewTag(langInput)

		fmt.Print("Enter viewer count: ")
		viewersInput, _ := reader.ReadString('\n')
		viewersInput = strings.ReplaceAll(viewersInput, "\n", "")
		viewers, _ := strconv.Atoi(viewersInput)

		msg, _ := b.Get(lang, raidMessage, bundle.Variant_Default)
		localized, _ := msg.Format(message.PlaceholderMap{
			"user":    message.NewUserID("1234"),
			"viewers": message.NewNumber(float64(viewers)),
		})

		printMessage(localized)
	}
}

func printMessage(fragments []message.Fragment) {
	for _, fragment := range fragments {
		if fragment.HasEmphasis() {
			fmt.Print(Bold)
		}

		switch f := fragment.(type) {
		case *message.TextFragment:
			fmt.Print(f.Value(), Reset)

		case *message.NumberFragment:
			fmt.Print(f.Value(), Reset)

		case *message.IDFragment:
			if f.Kind() == message.IDFragmentKind_User {
				fmt.Print("MrByte")
			}

			fmt.Print(Reset)
		}
	}

	fmt.Println()
}
