package bundle_test

import (
	"embed"
	"testing"

	"code.justin.tv/cplat/twitchling/localization/bundle"
	"code.justin.tv/cplat/twitchling/localization/language"
	"github.com/stretchr/testify/assert"
)

//go:embed test_data/valid/*
var validBundle embed.FS

//go:embed test_data/missing_language/*
var missingLanguageBundle embed.FS

//go:embed test_data/error_in_localization/*
var errorInLocalizationBundle embed.FS

func TestMessageBundle(t *testing.T) {
	assert.NotNil(t, validBundle)

	t.Run("should successfully load valid bundle without errors", func(t *testing.T) {
		b, err := bundle.New(validBundle)

		assert.NoError(t, err)
		assert.NotNil(t, b)
	})

	t.Run("returns partial bundle with error when a language is missing", func(t *testing.T) {
		b, err := bundle.New(missingLanguageBundle)

		assert.Error(t, err)
		assert.NotNil(t, b)

		m, err := b.Get(language.DefaultLanguage, bundle.MessageID("test"), bundle.Variant_Default)
		assert.NoError(t, err)
		assert.NotNil(t, m)

		lang, err := language.NewTag("en-GB")
		assert.NoError(t, err)

		m, err = b.Get(lang, bundle.MessageID("test"), bundle.Variant_Default)
		assert.Error(t, err)
		assert.Nil(t, m)
	})

	t.Run("returns partial bundle with error when a translation has issues", func(t *testing.T) {
		b, err := bundle.New(errorInLocalizationBundle)

		assert.Error(t, err)
		assert.NotNil(t, b)

		m, err := b.Get(language.DefaultLanguage, bundle.MessageID("test"), bundle.Variant_Default)
		assert.NoError(t, err)
		assert.NotNil(t, m)

		lang, err := language.NewTag("en-GB")
		assert.NoError(t, err)

		m, err = b.Get(lang, bundle.MessageID("test"), bundle.Variant_Default)
		assert.Error(t, err)
		assert.Nil(t, m)
	})
}
