package message

type Fragment interface {
	// Determines whether or not if a fragment has emphasis.
	HasEmphasis() bool

	// Returns a copy of the fragment with emphasis applied.
	WithEmphasis() Fragment

	// Only allow types in this package to implement the Fragment interface.
	fragment()
}

type TextFragment struct {
	value       string
	hasEmphasis bool
}

func (t *TextFragment) Value() string {
	return t.value
}

func (t *TextFragment) HasEmphasis() bool {
	return t.hasEmphasis
}

func (t *TextFragment) WithEmphasis() Fragment {
	return &TextFragment{t.value, true}
}

func (*TextFragment) fragment() {}

type IDFragmentKind byte

const (
	IDFragmentKind_User IDFragmentKind = iota
	IDFragmentKind_Emote
)

type IDFragment struct {
	kind        IDFragmentKind
	value       string
	hasEmphasis bool
}

func (i *IDFragment) Kind() IDFragmentKind {
	return i.kind
}

func (i *IDFragment) Value() string {
	return i.value
}

func (i *IDFragment) HasEmphasis() bool {
	return i.hasEmphasis
}

func (i *IDFragment) WithEmphasis() Fragment {
	return &IDFragment{i.kind, i.value, true}
}

func (*IDFragment) fragment() {}

type NumberFragmentKind byte

const (
	NumberFragmentKind_Integer NumberFragmentKind = iota
	NumberFragmentKind_Currency
	NumberFragmentKind_Percent
)

type NumberFragment struct {
	kind        NumberFragmentKind
	value       float64 // Use a float64 since it can represent all int32 values
	hasEmphasis bool
}

func (n *NumberFragment) Kind() NumberFragmentKind {
	return n.kind
}

func (n *NumberFragment) Value() float64 {
	return n.value
}

func (n *NumberFragment) HasEmphasis() bool {
	return n.hasEmphasis
}

func (n *NumberFragment) WithEmphasis() Fragment {
	return &NumberFragment{n.kind, n.value, true}
}

func (*NumberFragment) fragment() {}
