const Path = require('path');
const _ = require('lodash');
const wufoo = require('./wufoo');
const { exists, writeJson } = require('../utils');

const entries = function () {
  return wufoo.entries()
              .then((records) => {
                return records.map((record) => _.defaults(record, { status: 'pending' }));
              })
              .then((records) => {
                return _.keyBy(records, 'id');
              });
};

const load = function (file) {
  let module = Path.join('..', file);

  return exists(file)
            .then(() => require(module))
            .catch(() => {
              return {};
            });
};

const records = function (file) {
  // Load current data
  return load(file)
      // pull latest data from datastore
      .then((data) => {
        return entries().then((downloaded) => [ data, downloaded ])
      })
      // merge old and new data so we dont duplicate cards
      .then(([ data, downloaded ]) => {
        return _.defaults(data, downloaded);
      })
      // update the data file
      .then((data) => writeJson(file, data))
      // return only the items that dont have a status
      .then((data) => {
        // console.log(data);
        return _.values(data);
      });
};

module.exports = function (file) {
  const update = (record, properties) => {
    return load(file).then((data) => {
      data[record.id] = _.assign(record, properties);
      return data;
    })
    .then((data) => writeJson(file, data));
  };

  const filtered = (filter) => {
    return inProgress().then((records) => {
      return _.filter(records, filter);
    });
  };

  const inProgress = () => {
    return records(file).then((records) => {
      return _.reject(records, ['status', 'complete']);
    });
  };

  return {

    find (email) {
      return inProgress().then((records) => {
        return _.findLast(records, { email })
      })
      .then((record) => {
        if (!record) throw new Error('not_found');
        return record;
      });
    },

    pending () {
      return filtered(['status', 'pending']);
    },

    inReview (record) {
      console.log('GENERATED PROOF', record.email);
      return update(record, { status: 'in-review' });
    },

    finalized (record) {
      console.log('FINALIZE', record.email, record.quantity);
      return update(record, { status: 'complete', finalized: new Date() });
    },

    proofSent (record) {
      console.log('SENT PROOF', record.email);
      return update(record, { status: 'pending-approval', sent: new Date() });
    },

    reminderSent (record) {
      let reminder = record['reminder'] || [];

      reminder.push(new Date());

      console.log('SENT REMINDER', record.email, reminder.length);
      return update(record, { status: 'pending-approval', reminder });
    }
  }
};
