const Phone = require('google-libphonenumber');
const PhoneNumberFormat = Phone.PhoneNumberFormat;

// const phone = require('phone');
const RSVP = require('rsvp');
const request = require('request');
const key = process.env['WUFOO_KEY'];
const subdomain = process.env['WUFOO_SUBDOMAIN'];
const form = process.env['WUFOO_FORM'];
const endpoints = {
  fields: `https://${subdomain}.wufoo.com/api/v3/forms/${form}/fields.json`,
  entries: `https://${subdomain}.wufoo.com/api/v3/forms/${form}/entries.json`,
  entryCount: `https://${subdomain}.wufoo.com/api/v3/forms/${form}/entries/count.json`
};

const auth = {
  'user': key,
  'pass': 'twitch'
};

const qs = {
  pageSize: 100,
  pageStart: 0,
  sort: 'DateCreated'
};

const get = function () {
  return new RSVP.Promise((resolve, reject) => {
    request.get(...arguments, (error, response, body) => {
      try {
        if (error) return reject(error);
        resolve(JSON.parse(body));
      }
      catch (error) {
        console.error({ arguments, error, body })
        reject(error);
      }
    });
  });
};

const fieldMap = {
  'EntryId': 'id',
  'Field14': 'name',
  'Field7': 'email',
  'Field3': 'title',
  'Field8': 'phone',
  'Field10': 'twitch',
  'Field11': 'twitter',
  'Field12': 'skype',
  'Field9': 'quantity',
  'Field17': 'notes',
  'DateCreated': 'created'
};

const normalize = {
  name (key, value, obj) {
    return value;
  },
  email (key, value) {
    return value.toLowerCase();
  },
  title (key, value) {
    return value.toUpperCase();
  },
  created (key, value, obj) {
    return new Date(value);
  },
  twitter (key, value, obj) {
    let [ thandle, ...turl ] = (value || '').replace('@', '').split('/').reverse();
    return thandle || turl.join('');
  },
  phone (key, value, obj) {
    try {
      let phoneUtil = Phone.PhoneNumberUtil.getInstance();
      let phoneNumber = phoneUtil.parse(value, 'US');
      return phoneUtil.format(phoneNumber, PhoneNumberFormat.INTERNATIONAL);
    }
    catch (error) {
      console.error('phone', value, error.message);
      return value;
    };
  },
  quantity (key, value, obj) {
    return +value;
  }
};

const entry = function (obj) {
  return Object.keys(fieldMap).reduce((result, field) => {
    let key = fieldMap[field];
    let value = obj[field] || "";
    let normalizer = normalize[key];

    // strip whitespace, just incase
    value = value.trim && value.trim() || value;

    if ('function' == typeof normalizer) {
      value = normalizer(key, value, result);
    }

    result[key] = value;

    // console.log('entry', key, value);

    return result;
  }, {});
};

const entryCount = function () {
  return get(endpoints.entryCount, { auth }).then((r) => r.EntryCount);
};

const fields = function () {
  return get(endpoints.fields, { auth }).then((r) => r.Fields);
};

const entries = function () {
  return RSVP.hash({
      count: entryCount(),
      fields: fields()
    })
    .then(({ count, fields }) => {
      let { pageSize, pageStart, sort } = qs;
      let pages = Math.ceil(count / qs.pageSize);
      let requests = [];

      // console.log(fields);

      for(let page = 0; page < pages; page++) {
        let params = { pageStart: page * pageSize, pageSize, sort };
        let promise = get(endpoints.entries, { auth , qs: params }).then((r) => r.Entries);
        requests.push(promise);
      }
      return RSVP.all(requests);
    })
    .then((entries) => Array.prototype.concat.apply([], entries))
    .then((entries) => entries.map(entry));
};

exports.entries = function () {
  return entries().catch((error) => {
    console.error('entries', error);
    return [];
  });
};
