const Path = require('path');
const fs = require('fs');
const RSVP = require('rsvp');
const _glob = require('glob');
const crypto = require('crypto');

const chainPromises = function (items, chainable) {
  let promise = RSVP.Promise.resolve();
  return items.reduce((p, item) => {
    return p.then(() => chainable(item));
  }, promise);
};

const exists = function (file) {
  return new RSVP.Promise((resolve, reject) => {
    fs.access(file, (err) => {
      if (err) return reject(err);
      resolve();
    });
  });
};

const mkdirp = function (path) {
  const _mkdirp = require('mkdirp');
  return new RSVP.Promise((resolve, reject) => {
    _mkdirp(path, (error, result) => {
      if (error) return reject(error);
      resolve(result);
    });
  });
};

const move = function (source, destination) {
  return mkdirp(Path.dirname(destination)).then(() => {
    return fs.renameSync(source, destination);
  });
};

const write = function (filename, content) {
  return new RSVP.Promise((resolve, reject) => {
    fs.writeFile(filename, content, (error) => {
      if (error) return reject(error);
      resolve(content);
    });
  });
};

const writeJson = function (file, data) {
  return write(file, JSON.stringify(data, null, 2)).then(() => data);
};

const glob = function (pattern, options = {}) {
  return new RSVP.Promise((resolve, reject) => {
    _glob(pattern, options, function (er, files) {
      if (er) return reject(er);
      resolve(files);
    });
  });
};

const lookup = function (store) {
  return function (file) {
    let filename = Path.basename(file);

    // TODO: fixme just incase thres another - in the filename (expects 100-email@twitch.tv.pdf)
    let [ quantity, email ] = filename.replace(/\.pdf$/, '').split('-');

    if (/^_/.test(filename)) {
      return RSVP.Promise.resolve();
    }

    return store.find(email).catch(() => {
        let id = crypto.createHash('md5').update(filename).digest("hex");
        return { id, email, quantity };
      })
      .then((record) => {
        return {
          file,
          record
        };
      });
  };
};

const loadFolder = function (store, folder) {
  // load files in proof folder
  return glob("**/*.pdf", { cwd: folder })
    // add fullpath
    .then((files) => files.map((f) => Path.join(folder, f)))
    // lookup corresponding record
    .then((files) => RSVP.all(files.map(lookup(store))))
    // remove empty records
    .then((files) => files.filter((file) => !!file));
};

const send = require('./send');

const message = function (subject, template) {
  let email = '"Creative Services" <business-cards@creativeservices.twitch.tv>';
  let reallysend = true;
  return send(email, subject, template, reallysend);
};

const sendProof = message('Your Business Cards', 'proof-template.hbs');
const sendReminder = message('[REMINDER] Your Business Cards', 'reminder-template.hbs');

exports.glob = glob;
exports.chain = chainPromises;
exports.mkdirp = mkdirp;
exports.move = move;
exports.exists = exists;
exports.write = write;
exports.writeJson = writeJson;
exports.lookup = lookup;
exports.loadFolder = loadFolder;

// messaging
exports.send = send;
exports.sendProof = sendProof;
exports.sendReminder = sendReminder;
exports.message = message;