const GoogleSpreadsheet = require('google-spreadsheet');
const tmi = require('tmi.js')
const { bot, twitch, google } = require('./config.json');
const fetch = require('node-fetch');
const RSVP = require('rsvp');
const command = /^!ask\s/i;

const getAuthToken = ({ client_id, client_secret, scope }) => {
  return fetch(`https://id.twitch.tv/oauth2/token?client_id=${client_id}&client_secret=${client_secret}&scope=${scope}&grant_type=client_credentials`, {
    method: 'post'
  })
  .then((response) => response.json())
  .then((data) => {
    if (400 === data.status) throw new Error(data.message);
    return data;
  });
};

const addRow = (sheet, ...args) => {
  return new RSVP.Promise((resolve, reject) => {
    sheet.addRow(...args, (error) => {
      if(error) return reject(error);

      resolve();
    });
  });
};

const getMentions = (message) => {
  const matcher = /(@\w*)/g;
  const mentions = [];

  while (match = matcher.exec(message)) {
    let [, username] = match;
    console.log(username);
    mentions.push(username);
  }

  return mentions;
};

// Called every time a message comes in:
const onMessageHandler = (client, sheet, target, context, message, self) => {
  if (command.test(message)) {
    let mentions = getMentions(message);
    let question = message.replace(command, '')
    let payload = {
      Channel: target,
      Username: context.username,
      Question: question,
      Mentions: mentions.join(', ')
    };

    addRow(sheet, 1, payload)
      .then(() => console.log('asked', payload))
      .catch((error) => console.log('error', { error, payload }));

    // client.say(target, `@${context.username} thank you!`);
  }
}

// Called every time the bot connects to Twitch chat:
const onConnectedHandler = (client, addr, port) => {
  console.log(`* Connected to ${addr}:${port}`)

  sendInstructions(client);

  if (bot.announcement_minute_interval) {
    setInterval(() => sendInstructions(client), bot.announcement_minute_interval * 60 * 1000);
  }
}

const sendInstructions = (client) => {
  let { channels } = client.opts;

  if (bot.announce) {
    channels.forEach(channel => {
      client.say(channel, bot.announce);
    });
  }
};

// Called every time the bot disconnects from Twitch:
const onDisconnectedHandler = (reason) => { 
  console.log(`Disconnected: ${reason}`)
  process.exit(1)
}

const getSheet = ({ id, credentials }) => {
  return new RSVP.Promise((resolve, reject) => {
    // Create a document object using the ID of the spreadsheet - obtained from its URL.
    const doc = new GoogleSpreadsheet(id);

    // Authenticate with the Google Spreadsheets API.
    doc.useServiceAccountAuth(credentials, function (error) {
      if (error) return reject(error);
      resolve(doc);
    });
  });
};

return RSVP.hash({
  sheet: getSheet(google),
  // auth: getAuthToken(oauth)
})
.then(({ sheet }) => {
  // Create a client with our options:
  // let identity = { username: 'CreatorCampBot', password: `oauth:${auth.access_token}` };
  // let options = Object.assign({}, twitch, { identity });
  let client = new tmi.client(twitch)

  // Register our event handlers (defined below):
  client.on('message', onMessageHandler.bind(null, client, sheet));
  client.on('connected', onConnectedHandler.bind(null, client))
  client.on('disconnected', onDisconnectedHandler)

  // Connect to Twitch:
  client.connect();
})
.catch((error) => {
  console.error(error);
  process.exit(1);
});
