var fs = require('fs');

var apiRoot = __dirname + '/apis';
var serviceMap = null;
var serviceIdentifiers = [];
var serviceNames = [];

function buildServiceMap() {
  if (serviceMap !== null) return;

  // load info file for API metadata
  var infoFile = __dirname + '/metadata.json';
  serviceMap = JSON.parse(fs.readFileSync(infoFile).toString());

  var prefixMap = {};
  Object.keys(serviceMap).forEach(function(identifier) {
    serviceMap[identifier].prefix = serviceMap[identifier].prefix || identifier;
    prefixMap[serviceMap[identifier].prefix] = identifier;
  });

  fs.readdirSync(apiRoot).forEach(function (file) {
    var match = file.match(/^(.+?)-(\d+-\d+-\d+)\.api\.json$/);
    if (match) {
      var id = prefixMap[match[1]], version = match[2];
      if (serviceMap[id]) {
        serviceMap[id].versions = serviceMap[id].versions || [];
        serviceMap[id].versions.push(version);
      }
    }
  });

  Object.keys(serviceMap).forEach(function(identifier) {
    serviceMap[identifier].versions = serviceMap[identifier].versions.sort();
    serviceIdentifiers.push(identifier);
    serviceNames.push(serviceMap[identifier].name);
  });
}

function getServices() {
  buildServiceMap();
  return serviceIdentifiers;
}

function getServiceNames() {
  buildServiceMap();
  return serviceNames;
}

function serviceVersions(svc) {
  buildServiceMap();
  svc = serviceIdentifier(svc);
  return serviceMap[svc] ? serviceMap[svc].versions : null;
}

function serviceName(svc) {
  buildServiceMap();
  svc = serviceIdentifier(svc);
  return serviceMap[svc] ? serviceMap[svc].name : null;
}

function serviceFile(svc, version) {
  buildServiceMap();
  svc = serviceIdentifier(svc);
  if (!serviceMap[svc]) return null;

  var prefix = serviceMap[svc].prefix || svc;
  return apiRoot + '/' + prefix.toLowerCase() + '-' + version + '.api.json';
}

function paginatorsFile(svc, version) {
  buildServiceMap();
  svc = serviceIdentifier(svc);
  if (!serviceMap[svc]) return null;

  var prefix = serviceMap[svc].prefix || svc;
  return apiRoot + '/' + prefix + '-' + version + '.paginators.json';  
}

function waitersFile(svc, version) {
  buildServiceMap();
  svc = serviceIdentifier(svc);
  if (!serviceMap[svc]) return null;

  var prefix = serviceMap[svc].prefix || svc;
  return apiRoot + '/' + prefix + '-' + version + '.waiters.json';  
}

function load(svc, version) {
  buildServiceMap();
  svc = serviceIdentifier(svc);
  if (version === 'latest') version = null;
  version = version || serviceMap[svc].versions[serviceMap[svc].versions.length - 1];
  if (!serviceMap[svc]) return null;

  var api = JSON.parse(fs.readFileSync(serviceFile(svc, version)));

  // Try to load paginators
  if (fs.existsSync(paginatorsFile(svc, version))) {
    var paginators = JSON.parse(fs.readFileSync(paginatorsFile(svc, version)));
    api.paginators = paginators.pagination;    
  }

  // Try to load waiters
  if (fs.existsSync(waitersFile(svc, version))) {
    var waiters = JSON.parse(fs.readFileSync(waitersFile(svc, version)));
    api.waiters = waiters.waiters;    
  }

  return api;
}

function serviceIdentifier(svc) {
  return svc.toLowerCase();
}

module.exports = {
  serviceVersions: serviceVersions,
  serviceName: serviceName,
  serviceIdentifier: serviceIdentifier,
  serviceFile: serviceFile,
  load: load
};

Object.defineProperty(module.exports, 'services', {
  enumerable: true, get: getServices
});

Object.defineProperty(module.exports, 'serviceNames', {
  enumerable: true, get: getServiceNames
});
