import PubNub from 'pubnub';
import Controller from '@ember/controller';
import { sort } from '@ember/object/computed';
import { observer } from '@ember/object';
import { run } from '@ember/runloop';
import { task, timeout } from "ember-concurrency";
import { shuffle } from 'ember-composable-helpers/helpers/shuffle';

export default Controller.extend({

  init() {
    this._super(...arguments);
    this._connect();
    this.playedStreams = [];
  },

  rotationInterval: 15,

  sync: false,

  recycleStreams: false,

  showUnscheduled: false,

  getScheduledStreamers() {
    let { showUnscheduled } = this;

    return this.store.findAll('affiliate')
      .then((data) => data.filter((affiliate) => showUnscheduled || affiliate.scheduled));
  },

  getLiveStreams(scheduled) {
    let logins = scheduled.map(({ id }) => id);

    return this.store.query('stream', { logins })
      .then((data) => data.filter(({ online }) => online));
  },

  _queue: task(function * () {
    while (true) {
      let { rotationInterval, sync } = this;

      let displays = yield this.displays;
      let scheduledStreamers = yield this.getScheduledStreamers();
      let playingStreams = displays.map(({ stream }) => stream).compact();
      let liveStreams = yield this.getLiveStreams(scheduledStreamers.concat(playingStreams.map(({ login }) => ({ id: login }))));
      // let playedStreams = this.playedStreams || [];
      let availableDisplays = displays.filter(({ duration, playing, streamOnline }) => {
        console.log({ duration, rotationInterval, playing, streamOnline });
        return duration >= rotationInterval || !playing || !streamOnline;
      });

      // let streamsToRotateOut = availableDisplays.map(({ stream }) => stream).compact();
      // playedStreams.addObjects(streamsToRotateOut);

      this.setProperties({
        scheduledStreamers,
        liveStreams,
        // playedStreams
      });

      console.log('_queue', {
        scheduledStreamers,
        liveStreams,
        // playedStreams,
        availableDisplays,
        sync
      });

      if (sync) {
        let queue = liveStreams.reject((stream) => playingStreams.includes(stream));

        // recycle streams
        if (this.recycleStreams && queue.length < availableDisplays.length) {
          // make sure the stream is live before adding it bacl to the queue
          queue.addObjects(playingStreams.filter(({ online }) => online));
        }

        queue = shuffle(queue);

        availableDisplays.forEach((display, index) => {
          let stream = queue.objectAt(index);

          console.log('availableDisplay', { id: display.id, display, stream, queue });

          display.playStream(stream, true);
        });
      }

      // check everything every 30 seconds
      yield timeout(30 * 1000);
    }
  }).on('init').restartable(),

  displaySorting: Object.freeze(['id']),
  streamSorting: Object.freeze(['startAt']),

  displays: sort('model.displays', 'displaySorting'),

  uuid: 'server',

  _update(id, channel, state) {
    let attributes = state || {
      id,
      channel,
      online: true
    };

    attributes.socket = this.server;

    console.log('_update', { id, state });

    return this.store.push(this.store.normalize('display', attributes));
  },

  _remove({ uuid: id }) {
    let attributes = {
      id,
      channel: null,
      online: false,
      stream: null,
      socket: null
    };

    return this.store.push(this.store.normalize('display', attributes));
  },

  _connect() {
    this.server = new PubNub({
      publishKey: 'pub-c-0feaf8b0-e8ce-4587-95a6-f4eb03221890',
      subscribeKey: 'sub-c-4cd77d6c-d24e-11e8-b41d-e643bd6bdd68',
      presenceTimeout: 30,
      uuid: this.uuid
    });

    this.server.addListener({
      presence: (event) => {
        let { uuid, channel, state } = event;

        if (uuid === this.uuid) {
          return;
        }

        switch(event.action) {
        case 'state-change':
          run(this, '_update', uuid, channel, state);
          break;
        case 'join':
          run(this, '_update', uuid, channel, state);
          break;
        case 'timeout':
          run(this, '_remove', event);
          break;
        }
      }
    });

    this.server.subscribe({
      channels: [
        'tc.*'
      ]
    });

    this.server.hereNow({ includeUUIDs: true, includeState: true }, (status, { channels }) => {
      run(this, '_setDisplays', channels);
    });
  },

  _setDisplays(channels) {
    Object.keys(channels)
      .filter((name) => !/\.\*$/.test(name))
      .forEach((channel) => {
        let { occupants } = channels[channel];
        let occupant = occupants[0];
        let { uuid, state } = occupant;

        this._update(uuid, channel, state);
      });
  },

  scheduledStreamers: null,

  liveStreams: null,

  _resetQueue: observer('rotationInterval', 'showUnscheduled', function () {
    this.get('_queue').perform();
  }),

  actions: {

    play(display, stream) {
      display.playStream(stream, true);
    }

  }

});
