require './lib/s3'
class Video
  attr_accessor(
    :final_cut_path,
    :has_final_cut_video,
    :raw_video_path,
    :raw_subtitle_path,
    :has_transcoded_video,
    :has_transcoded_subtitles
  )

  def self.get_all_videos
    video_assets = BUCKET.objects.group_by { |obj| final_cut_path(obj.key) }
    video_assets.map {|final_cut_path, assets| from_assets(final_cut_path, assets)}
  end

  def self.final_cut_path(s3_path)
    return nil if s3_path.end_with?("/")
    [File.dirname(s3_path), File.basename(s3_path, ".*")] #strip extension
      .join("/")
      .gsub("video_processing/raw/", "")
      .gsub("video_processing/transcoded/", "")
      .concat(".mp4")
  end

  def self.get_raw_srt_path(raw_video_path)
    [File.dirname(raw_video_path), File.basename(raw_video_path, ".*")]
      .join("/")
      .concat(".srt")
  end

  def self.from_assets(final_cut_path, assets)
    video = Video.new
    video.final_cut_path = final_cut_path
    video.has_final_cut_video = false
    video.has_transcoded_video = false
    video.has_transcoded_subtitles = false
    assets.each do |asset|
      if asset.key.start_with?("video_processing/raw/")
        if ['.mpg', '.mov', '.avi', '.mp4', '.wmv', '.mxf'].any? { |ext| asset.key.end_with?(ext) }
          video.raw_video_path = asset.key
        elsif ['.scc', '.srt', '.sami', '.dfxp'].any? { |ext| asset.key.end_with?(ext) }
          video.raw_subtitle_path = asset.key
        end
      elsif asset.key.start_with?("video_processing/transcoded/")
        if asset.key.end_with?(".mp4")
          video.has_transcoded_video = true
        elsif asset.key.end_with?(".srt")
          video.has_transcoded_subtitles = true
        end
      else
        if ['.mp4'].any? { |ext| asset.key.end_with?(ext) }
          video.has_final_cut_video = true
        end
      end
    end
    video
  end

  def s3_url(method, path_name)
    SIGNER.presigned_url(
      method,
      key: send(path_name),
      bucket: BUCKET_NAME,
      expires_in: 604800
    )
  end

  def self.from_h(payload)
    video = Video.new
    video.final_cut_path = payload[:final_cut_path]
    video.has_final_cut_video = payload[:has_final_cut_video]
    video.raw_video_path = payload[:raw_video_path]
    video.raw_subtitle_path = payload[:raw_subtitle_path]
    video.has_transcoded_video = payload[:has_transcoded_video]
    video.has_transcoded_subtitles = payload[:has_transcoded_subtitles]
    video
  end

  def to_h
    {
      final_cut_path: final_cut_path,
      has_final_cut_video: has_final_cut_video,
      raw_video_path: raw_video_path,
      raw_subtitle_path: raw_subtitle_path,
      has_transcoded_video: has_transcoded_video,
      has_transcoded_subtitles: has_transcoded_subtitles
    }
  end

  def transcoded_video_path
    "video_processing/transcoded/#{final_cut_path}"
  end

  def transcoded_subtitle_path
    "video_processing/transcoded/#{final_cut_path}".gsub(".mp4", ".srt")
  end

  def needs_processing?
    !has_final_cut_video && (has_transcoded_video || raw_video_path)
  end

  def workflow_id
    Video.final_cut_path(@raw_video_path || @has_transcoded_video_path || @final_cut_path).gsub("/", "-")
  end

  def upload(method, file)
    raise 'File empty' if !File.size?(file.path)
    BUCKET.put_object(
      key: send(method),
      body: file
    )
  end

  def download(method, &blk)
    s3_key = send(method)
    ext = File.extname(s3_key)
    Tempfile.open(['file', ext]) do |f|
      BUCKET.object(s3_key).get(
        response_target: f.path
      )
      puts f.path
      blk.call(f)
      File.delete(f.path)
    end
  end
end
