defmodule Aggregator.Aggregation do
  use GenServer
  alias Aggregator.Scraper
  @aggregates %{
    "Creative" => %{
      game: "Creative",
      twitch_communities: MapSet.new([
		    "animation",
		    "cosplay",
		    "drawing",
		    "food",
		    "gamedev",
        "gamedevelopment",
        "indiedevelopers",
		    "glass",
		    "illustration",
		    "leatherwork",
		    "music",
		    "painting",
		    "pixelart",
		    "programming",
		    "sewing",
		    "graphite",
		    "perler",
		    "vector",
		    "figurines",
		    "woodworking",
		    "writing",
		    "sketch",
		    "watercolor",
		    "jewelry",
		    "electronics",
		    "3dmodeling",
		    "crossstitch",
		    "sculpting",
		    "editing",
		    "artstation",
		    "bodyart",
		    "fiberarts",
		    "metalworking",
		    "papercraft",
		    "design",
		    "digitalillustration",
		    "comedy",
		    "magic",
		    "dance",
		    "voiceacting",
		    "theater",
		    "robotics",
		    "chainmaille",
		    "brewing",
		    "virink",
		    "automotive",
		    "comics",
		    "pcbuilding",
		    "makeup",
        "photography",
        "crochet",
        "knitting",
        "paracord",
        "brickbuilding",
        "miniatures",
        "graphicdesign",
        "multicreative",
        "tattooing",
        "ludumdare",
        "tabletopminiatures",
        "musicproduction"
      ])
    }
  }

  def all_communities do
    @aggregates
    |> Map.values
    |> Enum.map(fn(aggregate) -> Map.get(aggregate, :twitch_communities) |> Enum.to_list end)
    |> List.flatten
    |>  Enum.into(MapSet.new())
  end

  def start_link(opts) do
    GenServer.start_link(__MODULE__, {generate_views(Scraper.fetch_from_cache)},  opts)
  end

  def update(scrape) do
     GenServer.call(__MODULE__, {:update, scrape})
  end

  def fetch(aggregate, limit, offset, sort, order, languages) do
     GenServer.call(__MODULE__, {:fetch, aggregate, limit, offset, sort, order, languages})
  end

  # Callbacks
  def handle_call({:update, scrape}, _from, _) do
    {:reply, :ok, {generate_views(scrape)}}
  end

  def handle_call({:fetch, aggregate, limit, offset, sort, order, languages}, _from, {streams_by_aggregate} = state) do
    case Map.get(@aggregates, aggregate) do
      nil ->  {:reply, nil, state}
      _ ->
        streams = get_in(streams_by_aggregate, [aggregate, :streams])
        |> Enum.sort_by(&(get_in(&1, [sort])))
        streams = case order do
          "desc" -> Enum.reverse(streams)
          _ -> streams
        end
        streams = filter_streams_languages(streams, languages)

        start = offset
        finish = (limit + (offset)) - 1
        aggregate = Map.get(@aggregates, aggregate, %{})
        |> Map.merge(%{streams: Enum.slice(streams, start..finish)})
        |> Map.merge(%{communities: get_in(streams_by_aggregate, [aggregate, :communities])})
        |> Map.put(:total, Enum.count(streams))
        {:reply, aggregate, state}
    end
  end

  def filter_streams_languages(streams, ""), do: streams
  def filter_streams_languages(streams, languages) do
    langs = String.split(languages, ",") |> Enum.into(MapSet.new)
    streams
    |> Enum.filter(fn(stream) -> MapSet.member?(langs, get_in(stream, ["channel", "broadcaster_language"])) end)
  end

  def generate_views(%{"streams" => streams, "communities" => communities} = scraped) do
    newViews = @aggregates
    |> Enum.map(fn({aggregate_name, _config}) ->
      {
        aggregate_name,
        %{
          streams: Enum.filter(streams, &(stream_matches?(aggregate_name, &1, scraped))),
          communities: Enum.filter(communities, &(community_matches?(aggregate_name, &1, scraped)))
        }
      }
    end)
    |> Enum.into(%{})
  end

  def stream_matches?("Creative", %{"channel" => %{"game" => "Creative"}}, _scraped), do: true
  def stream_matches?(aggregate, %{"community_id" => community_id}, %{"community_id_to_name" => community_id_to_name}) do

    community_name = Map.get(community_id_to_name, community_id)
    get_in(@aggregates, [aggregate, :twitch_communities])
    |>MapSet.member?(community_name)
  end
  def stream_matches?(aggregate, _, _), do: false

  def community_matches?(aggregate, %{"name" => name}, _scraped) do
    MapSet.member?(get_in(@aggregates, [aggregate, :twitch_communities]), String.downcase(name))
  end
  def community_matches?(_, _, _), do: false
end
