defmodule Aggregator.Scraper do
  defstruct communities: [], streams: [], community_id_to_name: %{}
  alias ExAws.S3
  alias Flow
  @per_page 100
  @bucket if Mix.env == :prod, do: "twitch-creative-assets", else: "twitch-creative-assets-dev"

  def scrape do
    %Aggregator.Scraper{}
    |> get_all_streams
    |> get_all_communities
    |> save_to_s3
  end

  def get_all_streams(scraper) do
    stream_count = HTTPoison.get!("https://api.twitch.tv/v5/streams?client_id=ajeucsszlbinfjp4j2o20pwvtvhdwzy").body
    |> Poison.decode!
    |> Map.get("_total")
    pages_to_fetch = round(stream_count / @per_page) + 1

    streams = 0..pages_to_fetch
    |> Flow.from_enumerable(stages: 20, min_demand: 0, max_demand: 1)
    |> Flow.partition
    |> Flow.flat_map(&get_streams/1)
    |> Enum.to_list
    |> Enum.uniq

    scraper
    |> Map.put(:streams, streams)
  end

  def get_all_communities(scraper) do
    response = HTTPoison.get!(Application.get_env(:aggregator, :comm_url)<>"/communities/top?client_id=ajeucsszlbinfjp4j2o20pwvtvhdwzy&limit=100").body
    |> Poison.decode!

    cursor = Map.get(response, "_cursor")
    communities = Map.get(response, "communities")
    get_all_communities(scraper, communities, cursor)
  end

  def get_all_communities(scraper, communities, "") do
    community_id_to_name = communities
    |> Enum.map(fn(community) -> {Map.get(community, "_id"), String.downcase(Map.get(community, "name"))} end)
    |> Enum.into(%{})

    communities = add_offline_communities(communities)

    scraper
    |> Map.put(:communities, communities)
    |> Map.put(:community_id_to_name, community_id_to_name)
  end
  def get_all_communities(scraper, communities, cursor) do
    response = HTTPoison.get!(Application.get_env(:aggregator, :comm_url)<>"/communities/top?client_id=ajeucsszlbinfjp4j2o20pwvtvhdwzy&limit=100&cursor=" <> cursor).body
    |> Poison.decode!

    cursor = Map.get(response, "_cursor")
    communities = communities
    |> List.flatten(Map.get(response, "communities"))

    get_all_communities(scraper, communities, cursor)
  end
  def get_all_communities(scraper, _, _), do: scraper

  def add_offline_communities(communities) do
    Aggregator.Aggregation.all_communities
    |> Enum.reduce(communities, fn(name, accum) ->
      case Enum.any?(communities, fn(community) -> String.downcase(Map.get(community, "name")) == name end) do
        true -> accum
        _ -> accum ++ [fetch_offline_community(name)]
      end
    end)
  end

  def fetch_offline_community(name) do
    response = HTTPoison.get!(Application.get_env(:aggregator, :comm_url)<>"/communities/?client_id=ajeucsszlbinfjp4j2o20pwvtvhdwzy&name=" <> name).body
    |> Poison.decode!
    |> Map.put("viewers", 0 )
    |> Map.put("channels", 0)
  end


  def save_to_s3(scraper) do
     S3.put_object(@bucket, "#{Mix.env}/aggregator/streams.json", Poison.encode!(scraper))
     |> ExAws.request!
  end

  def get_streams(offset, limit \\ @per_page) do
    HTTPoison.get!("https://api.twitch.tv/v5/streams?client_id=ajeucsszlbinfjp4j2o20pwvtvhdwzy&limit=#{limit}&offset=#{offset * limit}").body
      |> Poison.decode!
      |> Map.get("streams")
  end

  def fetch_from_cache do
    response = S3.get_object(@bucket, "#{Mix.env}/aggregator/streams.json")
    |> ExAws.request!
    response.body |> Poison.decode!
  end
end
