defmodule RacklaSkeleton.Router do
  @moduledoc false

  use Plug.Router
  require MappingToDsl
  import Rackla
  plug CORSPlug, headers: ["*"], origin: ~r{((^https?://)|^)(((.*\.)?twitch.tv)|(localhost)|(127\.0\.0\.1))(:[0-9]+)?$}

  plug Plug.Parsers, parsers: [:urlencoded, :json],
                       pass:  ["text/*"],
                       json_decoder: Poison
  plug :match
  plug :dispatch

  MappingToDsl.run [
    # %{
    #   route: "/api/timelapses/*_",
    #   url: "http://timelapser-prod.us-west-2.elasticbeanstalk.com"
    #   # url: "http://localhost:4000"
    # },
    %{
      route: "/api/aggregates/*_",
      url: Application.get_env(:urls, :aggregator_url)
    },
    %{
      route: "/api/channel_properties/*_",
      url: Application.get_env(:urls, :kv_store_url)
    },
  ]
  match _ do
    "Not Found"
    |> just
    |> response(compress: true)
  end

  def fetch_scopes(conn) do
    header_token = conn.req_headers |> Enum.into(%{}) |> Map.get("authorization")
    param_token = conn.params |> Map.get("oauth_token")

    (header_token || param_token)
    |> get_scopes_from_owl
  end

  def get_scopes_from_owl(nil), do: "null"
  def get_scopes_from_owl("OAuth " <> token), do: get_scopes_from_owl(token)
  def get_scopes_from_owl(token) do
    HTTPoison.get!("http://owl.prod.us-west2.justin.tv/validate\?token\=" <> token).body
  end

  def fetch_user("null"), do: "null"
  def fetch_user(scopes) do
    try do
      id = Poison.decode!(scopes)["owner_id"]
      user_service_host = Application.get_env(:urls, :user_service_host)

      HTTPoison.get!("http://users-service.#{user_service_host}.us-west2.twitch.tv/users?id=#{id}&prop=admin").body
      |> Poison.decode!
      |> Map.get("results")
      |> Enum.at(0)
      |> Poison.encode!
    rescue
      _ ->  "null"
    end
  end
end
