package sse

import (
	"bufio"
	"io"
	"strings"
)

// Event is a single event emitted by the text/event-stream endpoint.
type Event struct {
	Name string
	Data string
	buf  []string
}

func readLines(in io.Reader, out chan<- string, maxLen int) {
	defer close(out)
	rd := bufio.NewReader(in)
	for {
		line, _, err := rd.ReadLine()
		if err != nil {
			return
		}
		out <- string(line)
	}
}

func readEvents(in <-chan string, out chan<- *Event) {
	defer close(out)
	var ev *Event
	for line := range in {
		if ev == nil {
			ev = &Event{
				buf: make([]string, 0, 1),
			}
		}
		if line == "" {
			// blank line ... dispatch the event
			ev.Data = strings.Join(ev.buf, "\n")
			ev.buf = nil
			if ev.Data != "" {
				out <- ev
			}
			ev = nil
			continue
		}

		parts := strings.SplitN(line, ":", 2)
		name, value := parts[0], ""
		if len(parts) > 1 {
			value = parts[1]
		}
		if strings.HasPrefix(value, " ") {
			value = value[1:]
		}

		switch name {
		case "":
			// ignored (comment)
		case "event":
			ev.Name = value
		case "data":
			ev.buf = append(ev.buf, value)
		case "id":
			// ignored (unimplemented)
		case "retry":
			// ignored (unimplemented)
		default:
			// ignored (unknown)
		}
	}
}
