package sse

import (
	"errors"
	"net"
	"net/http"
	"time"

	"code.justin.tv/chat/tmi/netutil"
)

const (
	default_max_line_length = 16 * 1024
	default_dial_timeout    = 5 * time.Second
	default_data_timeout    = 20 * time.Second
)

var (
	ErrDisconnected = errors.New("connection closed")
)

// EventSource connects to the remote text/event-stream endpoint and allows
// its clients to receive individual events one at a time.
//
// A zeroed EventSource is not ready to use.
type EventSource struct {
	Target        string
	MaxLineLength int
	DialTimeout   time.Duration
	DataTimeout   time.Duration
	events        chan *Event
	resp          *http.Response
}

// NewEventSource creates a ready-to-use EventSource for a particular
// endpoint.
func NewEventSource(target string) (*EventSource, error) {
	return &EventSource{
		Target:        target,
		MaxLineLength: default_max_line_length,
		DialTimeout:   default_dial_timeout,
		DataTimeout:   default_data_timeout,
	}, nil
}

// Connect establishes the connection to the remote endpoint.
// It returns after reading the http headers.
func (T *EventSource) Connect() error {
	// build the client
	dialer, ch := buildDialer(T.DialTimeout, T.DataTimeout)
	tr := &http.Transport{
		Dial:              dialer,
		DisableKeepAlives: true,
	}
	client := &http.Client{Transport: tr}

	// build the request
	req, err := http.NewRequest("GET", T.Target, nil)
	if err != nil {
		return err
	}
	req.Header.Set("Accept", "text/event-stream")
	req.Header.Set("Cache-Control", "no-cache")
	req.Header.Set("Last-Event-ID", "")

	// make the request!
	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	T.resp = resp

	// the connection is established!
	// it's now up to the caller to close the connection when they're done

	// prepare to read from the connection...
	lines := make(chan string)
	maxLen := T.MaxLineLength
	if maxLen <= 0 || maxLen > default_max_line_length {
		maxLen = default_max_line_length
	}
	go readLines(resp.Body, lines, maxLen)

	// read from the connection
	conn := <-ch

	foo := make(chan string, 10)

	go func() {
		defer close(foo)
		for line := range lines {
			foo <- line
			err := conn.SetDeadline(time.Now().Add(T.DataTimeout))
			if err != nil {
				return
			}
		}
	}()

	// We close the channel if we disconnect, so making the channel
	// here (intead of in NewEventSource) allows Connect() to reconnect
	T.events = make(chan *Event, 1)
	go readEvents(foo, T.events)

	return nil
}

func (T *EventSource) Close() error {
	return T.resp.Body.Close()
}

// Get returns a single Event from the stream, blocking until one
// is available or the connection closes.
func (T *EventSource) Get() (*Event, error) {
	ev, ok := <-T.events
	if !ok {
		return nil, ErrDisconnected
	}
	return ev, nil
}

func buildDialer(dialTimeout, dataTimeout time.Duration) (func(nw, addr string) (net.Conn, error), chan net.Conn) {
	ch := make(chan net.Conn, 1)

	baseDialer := netutil.BuildDialer(&dialTimeout, &dataTimeout)
	dialer := func(nw, addr string) (net.Conn, error) {
		c, err := baseDialer(nw, addr)
		if err != nil {
			return nil, err
		}

		select {
		case ch <- c:
		default:
			c.Close()
			return nil, errors.New("can't send connection")
		}
		return c, nil
	}

	return dialer, ch
}
