package smtp

import (
	"errors"
	"fmt"
	"io"
	"log"
	"net"
	"net/smtp"
	"strconv"
	"time"
)

// Client manages the SMTP connection and sends messages in queued order.
// Client is safe to use from multiple goroutines.
type Client struct {
	client    *smtp.Client
	addr      string
	jobQueue  chan *queueJob
	closeChan chan bool
}

type queueJob struct {
	message *Message
	errChan chan error
	ttl     int
}

const (
	jobQueueSize    = 10
	jobInitialTTL   = 3
	jobQueueTimeout = 1000
)

// NewClient returns a Client that connects to hostname:port.
func NewClient(hostname string, port int) (*Client, error) {
	c := &Client{}
	c.addr = net.JoinHostPort(hostname, strconv.Itoa(port))
	c.jobQueue = make(chan *queueJob, jobQueueSize)
	c.closeChan = make(chan bool)
	go c.processJobQueue()
	return c, nil
}

// NewMail returns a Message.
func (c *Client) NewMail() *Message {
	m := c.newMessage()
	return m
}

// Close waits for the message queue to finish sending and closes the SMTP
// connection.
func (c *Client) Close() error {
	close(c.jobQueue)
	<-c.closeChan
	return c.client.Close()
}

func (c *Client) queueMessage(message *Message) error {
	ch := make(chan bool)
	go func() {
		errChan := make(chan error)
		c.jobQueue <- &queueJob{message, errChan, jobInitialTTL}
		ch <- true
	}()
	select {
	case <-ch:
		return nil
	case <-time.After(time.Millisecond * jobQueueTimeout):
		return errors.New("smtp: Send message queue is full")
	}
}

func (c *Client) processJobQueue() {
	var err error
	for job := range c.jobQueue {
		for c.client == nil {
			c.client, err = c.dialServer()
			if err == nil {
				break
			}
			log.Printf("smtp: error dialing server: %v", err)
			// TODO: implement exponential backoff
			retryTime := time.Millisecond * 500
			time.Sleep(retryTime)
		}
		c.processJob(job)
	}
	c.closeChan <- true
}

func (c *Client) processJob(job *queueJob) error {
	var err error
	message := job.message
	if err = message.valid(); err != nil {
		return fmt.Errorf("smtp: invalid message: %v", err)
	}
	defer func() {
		if err != nil {
			c.requeueJob(job)
		}
	}()
	err = c.client.Mail(message.from)
	if err != nil {
		return err
	}

	for _, rcpt := range message.to {
		if err = c.client.Rcpt(rcpt); err != nil {
			return err
		}
	}
	var writer io.WriteCloser
	if writer, err = c.client.Data(); err != nil {
		return err
	}
	if err = message.writeDataTo(writer); err != nil {
		return err
	}
	if err = writer.Close(); err != nil {
		return err
	}
	return nil
}

func (c *Client) dialServer() (newClient *smtp.Client, err error) {
	newClient, err = smtp.Dial(c.addr)
	return
}

func (c *Client) requeueJob(job *queueJob) {
	c.client.Close()
	c.client = nil
	job.ttl--
	if job.ttl > 0 {
		c.jobQueue <- job
		// TODO: implement exponential backoff
		retryTime := time.Millisecond * 500
		time.Sleep(retryTime)
	}
}
