package smtp

import(
  "bytes"
  "encoding/hex"
  "errors"
  "fmt"
  "hash/crc32"
  "io"
  "os"
  "regexp"
  "strings"
  "time"
)

// Message
type Message struct {
  client *Client
  from string
  to []string
  headers []header
  buffer *bytes.Buffer
  reader *bytes.Reader
}

type header struct {
  field string
  body string
}

const(
  bufferInitialLength = 4096
)

var (
  crlfRegexp *regexp.Regexp
)

func init() {
  crlfRegexp, _ = regexp.Compile("(\r\n)|\r|\n")
}

func (client *Client) newMessage() *Message {
  m := &Message{
    client: client,
    to: []string{},
    headers: []header{},
    buffer: bytes.NewBuffer(make([]byte, 0, bufferInitialLength)),
  }
  origDate := time.Now()
  m.Header("date", origDate.Format(time.RubyDate))
  return m
}

// Header adds a header field to the message.
func (m *Message) Header(field, body string) error {
  // TODO: check header field syntax and occurence limits
  field = strings.Title(field)
  m.headers = append(m.headers, header{field, body})
  switch field {
  case "From":
    m.from = body
  case "To", "Cc", "Bcc":
    addrs :=  strings.Split(body, ",")
    for i, addr := range addrs {
      addrs[i] = strings.Trim(addr, " ")
    }
    m.to = append(m.to, addrs...)
  }
  return nil
}

// Write implements the io.writer interface, and writes to the body of the
// message. Converts CR/LF to CRLF.
func (m *Message) Write(p []byte) (n int, err error) {
  if m.buffer == nil {
    return 0, errors.New("message is closed")
  }
  return m.buffer.Write(crlfRegexp.ReplaceAllLiteral(p, []byte("\r\n")))
}

// Close implements the io.closer interface, and queues the message to be sent.
func (m *Message) Close() error {
  m.reader = bytes.NewReader(m.buffer.Bytes())
  m.buffer = nil
  err := m.client.queueMessage(m)
  return err
}

func (m *Message) valid() error {
  if m.from == "" {
    return errors.New("no from address")
  }
  if len(m.to) == 0 {
    return errors.New("no recipients")
  }
  return nil
}

func (m *Message) messageId() (string, error) {
  hash := crc32.New(crc32.MakeTable(crc32.IEEE))  // crc32 IEEE to calculate message id
  timestamp := time.Now().Unix()
  hostname, _ := os.Hostname()
  for _, header := range m.headers {
    if _, err := fmt.Fprintf(hash, "%s: %s\r\n", header.field, header.body); err != nil {
      return "", fmt.Errorf("error calculating message id: %v", header.field, err)
    }
  }
  if _, err := m.reader.Seek(0,0); err != nil {
    return "", fmt.Errorf("error calculating message id: %v", err)
  }
  if _, err := m.reader.WriteTo(hash); err != nil {
    return "", fmt.Errorf("error calculating message id: %v", err)
  }
  return fmt.Sprintf("<%s.%d@%s>", hex.EncodeToString(hash.Sum(nil)), timestamp, hostname), nil
}

func (m *Message) writeDataTo(writer io.Writer) error {
  // TODO: enforce RFC 2822 max line length
  messageId, err := m.messageId()
  if err != nil {
    return err
  }
  m.Header("message-id", messageId)

  for _, header := range m.headers {
    if _, err = fmt.Fprintf(writer, "%s: %s\r\n", header.field, header.body); err != nil {
      return fmt.Errorf("smtp: error writing message: %v", header.field, err)
    }
  }

  if _, err = fmt.Fprint(writer, "\r\n"); err != nil {
    return fmt.Errorf("smtp: error writing message: %v", err)
  }
  if _, err := m.reader.Seek(0,0); err != nil {
    return fmt.Errorf("error writing message: %v", err)
  }
  if _, err := m.reader.WriteTo(writer); err != nil {
    return fmt.Errorf("error writing message: %v", err)
  }
  if _, err = fmt.Fprint(writer, "\r\n"); err != nil {
    return fmt.Errorf("error writing message:: %v", err)
  }
  return nil
}
