package owl

import (
	"encoding/json"
	"fmt"
	"net/url"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/web/owl/oauth2"
	"golang.org/x/net/context"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "owl"
)

type Client struct {
	twitchhttp.Client
}

type ErrorResponse struct {
	Status  int    `json:"status"`
	Message string `json:"message"`
	Error   string `json:"error"`
}

func NewClient(conf twitchhttp.ClientConf) (*Client, error) {
	confWithXactName := addXactNameIfNoneExists(conf)
	twitchClient, err := twitchhttp.NewClient(confWithXactName)
	if err != nil {
		return nil, err
	}

	return &Client{twitchClient}, nil
}

func addXactNameIfNoneExists(conf twitchhttp.ClientConf) twitchhttp.ClientConf {
	if conf.TimingXactName != "" {
		return conf
	}
	conf.TimingXactName = defaultTimingXactName
	return conf
}

// Validate makes an API call to Owl to lookup the authorization for a token.
// If the token is invalid or doesn't have a superset of scopes, returns an authorziation with Valid=false
func (c *Client) Validate(ctx context.Context, token string, scopes []string, reqOpts *twitchhttp.ReqOpts) (*oauth2.Authorization, error) {
	scope := oauth2.BuildScope(scopes)
	query := url.Values{
		"token": {token},
	}
	if len(scope) > 0 {
		query.Add("scope", scope)
	}

	req, err := c.NewRequest("GET", "/validate?"+query.Encode(), nil)

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.validate",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()

	if resp.StatusCode >= 400 {
		var e ErrorResponse
		json.NewDecoder(resp.Body).Decode(&e)

		return nil, fmt.Errorf(`response code %v, "%v"`, resp.StatusCode, e.Message)
	}

	var o oauth2.Authorization
	err = json.NewDecoder(resp.Body).Decode(&o)
	if err != nil {
		return nil, err
	}

	return &o, nil
}
