package oauth2

import (
	"crypto/rand"
	"errors"
	"fmt"
	"math/big"
	"strings"
	"time"
)

// Authorization represents an OAuth2 Authorization
// While owner ids are integers at present, we should support
// more advanced "GUID-style" identifiers that may indicate information
// like sharding references or site association
type Authorization struct {
	Valid     bool      `json:"valid"`
	ID        int       `json:"_id"`
	Scope     []string  `json:"scope"`
	OwnerID   string    `json:"owner_id"`
	ClientID  string    `json:"client_id"`
	CreatedAt time.Time `json:"created_at"`
	UpdatedAt time.Time `json:"updated_at"`
	UUID      string    `json:",omitempty"`
	Hashes    []string  `json:"-"`
}

// Token is another view into an Authorization for endpoints concerned
// with the actual token values
type Token struct {
	AccessToken     string
	AccessTokenHash string
	RefreshToken    string
	Scope           []string
	Code            string
	RowID           string
	// Expires          time.Time (unimplemented)
}

type Client struct {
	ID                   string
	ClientID             string
	RedirectURI          string
	PasswordGrantAllowed bool
}

type ResponseType int

const (
	TokenResponse ResponseType = iota + 1
	CodeResponse
)

var (
	ErrNoAuth = fmt.Errorf("oauth2: no authorization found")
	// InvalidAuth represents that no authorization was found for a token
	InvalidAuth = &Authorization{Valid: false}
)

const tokenLength = 30
const alphabet = "abcdefghijklmnopqrstuvwxyz0123456789"

var alphabetLength = big.NewInt(int64(len(alphabet)))

func randomToken() (string, error) {
	var bytes = make([]byte, tokenLength)

	for i := range bytes {
		offset, err := rand.Int(rand.Reader, alphabetLength)
		if err != nil {
			return "", err
		}
		bytes[i] = alphabet[offset.Uint64()]
	}

	return string(bytes), nil
}

func NewCode() (string, error) {
	return randomToken()
}

func NewToken() (*Token, error) {
	token := new(Token)
	var err error

	if err = token.ResetAccessToken(); err != nil {
		return nil, errors.New("could not generate access token")
	}

	return token, nil
}

func (T *Token) ResetAccessToken() error {
	accessToken, err := randomToken()
	if err != nil {
		return err
	}

	T.AccessToken = accessToken
	T.AccessTokenHash = Hashify(T.AccessToken)
	return nil
}

// ParseScopes takes a space separated string of scopes and returns an array of scopes,
// used for parsing scopes from HTTP params or the DB.
func ParseScope(scope string) []string {
	if scope == "" {
		return []string{}
	}
	return DedupeSlice(strings.Split(scope, " "))
}

// BuildScope takes an arary of scopes and returns a space separated string of scopes so
// they can be used in an HTTP response or stored in the DB.
func BuildScope(scope []string) string {
	return strings.Join(DedupeSlice(scope), " ")
}

// contains checks whether requestedScopes is a subset of authScopes
func contains(authScopes, requestedScopes []string) bool {
	scopeMap := make(map[string]struct{})
	for _, scope := range authScopes {
		scopeMap[scope] = struct{}{}
	}

	for _, scope := range requestedScopes {
		if _, ok := scopeMap[scope]; !ok && scope != "" {
			return false
		}
	}
	return true
}

func DedupeSlice(l []string) []string {
	dedupedSlice := make([]string, 0)
	deduper := make(map[string]bool)
	for _, str := range l {
		if str != "" && !deduper[str] {
			deduper[str] = true
			dedupedSlice = append(dedupedSlice, str)
		}
	}
	return dedupedSlice
}
