package oauth2

import (
	"bytes"
	"crypto/hmac"
	"crypto/sha256"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"strings"
)

type RefreshTokenData struct {
	RowID string `json:"_id"`
	UUID  string `json:"_uuid"`
}

var (
	ErrInvalidRefreshToken = fmt.Errorf("oauth2: invalid refresh token")
	ErrMissingHMACKey      = fmt.Errorf("oauth2: missing HMAC key")
)

const tokenSeparator = "%"

// generateRefreshToken creates a signed refresh token given the
// SQL row ID and CQL UUID for the authorization.
// The format of the refresh token is:
// <json body (base64)>%<hmac sig (base64)>
// To verify, we decode the Json body and verify the HMAC signature
func (T *Token) GenerateRefreshToken(refreshTokenData *RefreshTokenData, key []byte) error {
	if len(key) == 0 {
		return ErrMissingHMACKey
	}

	jsonBody, err := json.Marshal(refreshTokenData)
	if err != nil {
		return err
	}

	var buffer bytes.Buffer
	encoder := base64.NewEncoder(base64.StdEncoding, &buffer)
	encoder.Write(jsonBody)
	encoder.Close()

	// Calculate hash of the base64 encoded body
	mac := hmac.New(sha256.New, key)
	mac.Write(buffer.Bytes())

	buffer.WriteString(tokenSeparator)

	encoder.Write(mac.Sum(nil))
	encoder.Close()

	T.RefreshToken = buffer.String()
	return nil
}

// DecodeRefreshToken decodes a token created by GenerateRefreshToken. It returns the sql and cql
// authorization ids.
func DecodeRefreshToken(token string, key []byte) (*RefreshTokenData, error) {
	if len(key) == 0 {
		return nil, ErrMissingHMACKey
	}

	parts := strings.SplitN(token, tokenSeparator, 2)

	if len(parts) != 2 {
		return nil, ErrInvalidRefreshToken
	}

	mac := hmac.New(sha256.New, key)
	mac.Write([]byte(parts[0]))
	expectedMAC := mac.Sum(nil)

	messageMAC, err := base64.StdEncoding.DecodeString(parts[1])
	if err != nil {
		return nil, ErrInvalidRefreshToken
	}

	if !hmac.Equal(messageMAC, expectedMAC) {
		return nil, ErrInvalidRefreshToken
	}

	jsonBody, err := base64.StdEncoding.DecodeString(parts[0])
	if err != nil {
		return nil, ErrInvalidRefreshToken
	}

	var message RefreshTokenData
	if err := json.Unmarshal(jsonBody, &message); err != nil {
		return nil, ErrInvalidRefreshToken
	}

	if len(message.RowID) == 0 {
		return nil, ErrInvalidRefreshToken
	}

	return &message, nil
}
