package oauth2

// WhitelistClients is a list of client ids whose tokens we will automatically upgrade.
// We need to be able to grant additional scopes to internal clients, such as iOS, so they can use
// APIs with scopes created after authorizing. If one of these clients make a request missing
// whitelisted scopes, we'll automatically add them
var WhitelistClients = []string{
	"5880",  // iphone
	"11769", // ipad
	"17693", // android
	"20094", // site
	"8187",  // Drew McLean test app
}

// WhitelistScopes is a list of scopes we'll automatically add to whitelisted clients
var WhitelistScopes = []string{
	"user_friends_edit",
	"user_friends_read",
	"user_presence_edit",
	"user_presence_friends_read",
	"channel_feed_edit",
	"channel_feed_read",
	"channel_feed_report",
	"user_subscriptions_edit",
	"channel_editor",
}

// AugmentScope adds Whitelisted scopes into the Authorization if the grant is for a Whitelisted Client
func AugmentScope(auth *Authorization) *Authorization {
	if auth == nil || auth == InvalidAuth {
		return InvalidAuth
	}

	if !contains(WhitelistClients, []string{auth.ClientID}) {
		return auth
	}

	// Whitelisted client, implicitly grant whitelistedscopes.
	for _, scope := range WhitelistScopes {
		if !contains(auth.Scope, []string{scope}) {
			auth.Scope = append(auth.Scope, scope)
		}
	}

	return auth
}

// VerifyScope returns the authorization if it has sufficient scopes. Otherwise returns InvalidAuth
func VerifyScope(auth *Authorization, scopes []string) *Authorization {
	if auth == nil || auth == InvalidAuth || !contains(auth.Scope, scopes) {
		return InvalidAuth
	}

	return auth
}
