package api

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"strings"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stvp/rollbar"

	"goji.io"
	"goji.io/pat"
)

// AllowedGames is a mapping of allowed games
var AllowedGames = map[string]bool{
	"creative": true,
	"all":      true,
}

// Server is cool
type Server struct {
	*goji.Mux

	Stats statsd.Statter
}

// NewServer handles routing from endpoint paths to handler functions
func NewServer(stats statsd.Statter) (*Server, error) {
	server := twitchhttp.NewServer()
	s := &Server{
		server,
		stats,
	}

	s.UseC(stripTrailingSlash)
	s.UseC(addCors)

	// Private
	s.HandleFuncC(pat.Post("/user/communities"), s.createHandler(s.setUserCommunity, "post_user_communities"))
	s.HandleFuncC(pat.Get("/user/communities"), s.createHandler(s.getUserCommunity, "get_user_communities"))

	// Edge
	s.HandleFuncC(pat.Get("/games/:game/communities"), s.createHandler(s.communityList, "games-{{game}}-communities"))
	s.HandleFuncC(pat.Get("/games/:game/communities/:community"), s.createHandler(s.communityDetail, "games-{{game}}-communities-{{community}}"))
	s.HandleFuncC(pat.Put("/games/:game/communities/:community"), s.createHandler(s.communityUpdate, "set-games-{{game}}-communities-{{community}}"))
	s.HandleFuncC(pat.Post("/games/:game/communities"), s.createHandler(s.communityProposal, "community_proposal"))
	s.HandleFuncC(pat.Post("/games/:game/communities/banner"), s.createHandler(s.bannerUpload, "banner_upload"))
	s.HandleFuncC(pat.Get("/streams/communities/:community"), s.createHandler(s.communityStreams, "streams-communities-{{community}}"))
	s.HandleFuncC(pat.Get("/streams/communities/:community/with_game"), s.createHandler(s.communityAndGameStreams, "streams-communities-with_game-{{community}}"))
	s.HandleFuncC(pat.Get("/streams/communities/:room/chats"), s.createHandler(s.getChats, "room_chats"))

	return s, nil
}

// createHandler logs the request information. Includes timing but not response status.
func (s *Server) createHandler(handler func(context.Context, http.ResponseWriter, *http.Request), statName string) func(context.Context, http.ResponseWriter, *http.Request) {
	return func(ctx context.Context, w http.ResponseWriter, r *http.Request) {
		start := time.Now()
		endpointCtx := context.WithValue(ctx, "endpoint", statName)
		handler(endpointCtx, w, r)
		duration := time.Since(start)
		_ = s.Stats.Inc(fmt.Sprintf("endpoints.%s", statName), 1, 0.1)
		_ = s.Stats.TimingDuration(fmt.Sprintf("endpoints.%s", statName), duration, 0.1)

		logString := fmt.Sprintf("%s - [%s] \"%s %s %s\" (%dms)",
			strings.Split(r.RemoteAddr, ":")[0],
			start.Format("02/Jan/2006:15:04:05 -0700"),
			r.Method,
			r.URL.RequestURI(),
			r.Proto,
			duration/time.Millisecond,
		)
		log.Printf(logString)
	}
}

// stripTrailingSlash will redirect requests to /endpoint/ to /endpoint with a 301
func stripTrailingSlash(h goji.Handler) goji.Handler {
	return goji.HandlerFunc(func(ctx context.Context, w http.ResponseWriter, r *http.Request) {
		path := r.URL.Path
		if path[len(path)-1:] == "/" {
			r.URL.Path = path[:len(path)-1]
			http.Redirect(w, r, r.URL.String(), http.StatusMovedPermanently)
			return
		}
		h.ServeHTTPC(ctx, w, r)
	})
}

// addCors wraps all endpoints in jsonp
func addCors(h goji.Handler) goji.Handler {
	return goji.HandlerFunc(func(ctx context.Context, w http.ResponseWriter, r *http.Request) {
		w.Header().Set("Access-Control-Allow-Origin", "*")
		w.Header().Set("Access-Control-Allow-Credentials", "false")
		w.Header().Set("Access-Control-Allow-Headers", "Accept, Authorization, Client-ID, Twitch-Api-Token, X-Forwarded-Proto, X-Requested-With")

		h.ServeHTTPC(ctx, w, r)
	})
}

type errorMessage struct {
	Message string `json:"message"`
	Status  int    `json:"status"`
	Error   string `json:"error"`
}

// ServeError will serve HTTP errors from a status code
func ServeError(w http.ResponseWriter, r *http.Request, msg string, code int) {
	w.WriteHeader(code)
	err := json.NewEncoder(w).Encode(&errorMessage{
		Message: msg,
		Status:  code,
		Error:   http.StatusText(code),
	})
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
	}
}
