package api

import (
	"encoding/json"
	"fmt"
	"image"
	"math/rand"
	"mime/multipart"
	"time"
	// Package image/jpeg is not used explicitly in the code below
	_ "image/jpeg"
	"net/http"
	"strconv"
	"strings"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/creative/communities/lib"
	"code.justin.tv/creative/communities/lib/orm"
	"code.justin.tv/creative/communities/lib/owlclient"
	"code.justin.tv/creative/communities/lib/s3client"
	"code.justin.tv/creative/communities/lib/smtpclient"
	"code.justin.tv/creative/communities/lib/twitchapi"
	"code.justin.tv/creative/communities/models"
	"code.justin.tv/creative/communities/settings"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/jinzhu/gorm"
	"github.com/stvp/rollbar"
	"goji.io/pat"
	"golang.org/x/net/context"
)

const (
	streams      = "streams"
	viewers      = "viewers"
	frontPage    = "fp"
	maxLimit     = 100
	defaultLimit = 20
)

// CommunityResponse represents the endpoint response for a single Community
type CommunityResponse struct {
	Name            string  `json:"name"`
	Game            string  `json:"game"`
	Banner          string  `json:"banner"`
	StreamCount     int     `json:"streams"`
	ViewerCount     int     `json:"viewers"`
	PromotedChannel *string `json:"promoted_channel"`
}

// CommunityDetailResponse represents the Community Detail endpoint response
type CommunityDetailResponse struct {
	Links     CommunityListLinks `json:"_links"`
	Community CommunityResponse  `json:"community"`
}

// CommunityListResponse represents a Community List endpoint response
type CommunityListResponse struct {
	Total       int                 `json:"_total"`
	Links       CommunityListLinks  `json:"_links"`
	Communities []CommunityResponse `json:"communities"`
}

// CommunityListLinks represents links within response
type CommunityListLinks struct {
	Self string `json:"self"`
}

// BannerUploadResponse contains the response to uploading a banner
type BannerUploadResponse struct {
	BannerURL string `json:"banner"`
}

// CommunityProposalResponse contains the response to a community proposal
type CommunityProposalResponse struct {
	Community CommunityResponse `json:"community"`
}

const (
	bannerHeight = 900
	bannerWidth  = 1600
)

// parseOAuth parses an HTTP oAuth Authentication string.
func parseOAuth(auth string) string {
	const prefix = "OAuth "
	if !strings.HasPrefix(auth, prefix) {
		return ""
	}
	return auth[len(prefix):]
}

func currentUser(ctx context.Context, r *http.Request) (*twitchapi.UserServiceUser, error) {
	token := parseOAuth(r.Header.Get("Authorization"))
	if token == "" {
		return nil, ErrOauthTokenNotProvided
	}
	owlClient, err := owlclient.Client()
	if err != nil {
		return nil, twitchapi.ErrUnverifiedUser
	}
	scopes := []string{}
	reqOpts := twitchhttp.ReqOpts{}
	auth, err := owlClient.Validate(ctx, token, scopes, &reqOpts)
	if err != nil || !auth.Valid {
		return nil, twitchapi.ErrUnverifiedUser
	}
	userID := auth.OwnerID
	users, err := twitchapi.GetUserProperties(ctx, []string{"email_verified", "email"}, []string{userID}, nil)
	if err != nil || len(users) == 0 {
		return nil, twitchapi.ErrUnverifiedUser
	}
	user := users[0]
	return &user, nil
}

func s3URL(key string) string {
	return fmt.Sprintf("https://s3-us-west-2.amazonaws.com/%s/%s", settings.Resolve("creativeAssetsBucket"), key)
}

func uploadImageToCommunitiesBucket(key string, file multipart.File) error {
	assetsClient := s3client.Client()
	_, err := assetsClient.PutObject(&s3.PutObjectInput{
		Bucket: aws.String(settings.Resolve("creativeAssetsBucket")),
		Key:    aws.String(key),
		ACL:    aws.String("public-read"),
		Body:   file,
	})
	return err
}

func copyImageToCommunitiesBucket(source, destination string) error {
	assetsClient := s3client.Client()
	_, err := assetsClient.CopyObject(&s3.CopyObjectInput{
		Bucket:     aws.String(settings.Resolve("creativeAssetsBucket")),
		Key:        aws.String(destination),
		CopySource: aws.String(source),
		ACL:        aws.String("public-read"),
	})
	return err
}

func validateBanner(r *http.Request, file multipart.File) error {
	banner, _, err := image.Decode(file)
	if err != nil {
		rollbar.RequestError(rollbar.WARN, r, err)
		return ErrBannerNotUploaded
	}
	if banner.Bounds().Max.X != bannerWidth || banner.Bounds().Max.Y != bannerHeight {
		return ErrWrongBannerDimensions
	}
	_, err = file.Seek(0, 0)
	if err != nil {
		rollbar.RequestError(rollbar.WARN, r, err)
		return ErrBannerNotUploaded
	}
	return nil
}

func publishBanner(user twitchapi.UserServiceUser, community, banner string) (*string, error) {
	bannerPrefix := "https://s3-us-west-2.amazonaws.com/" + settings.Resolve("creativeAssetsBucket")
	if !strings.HasPrefix(banner, bannerPrefix) {
		return nil, ErrPreviewImageNotUploaded
	}

	s3Key := fmt.Sprintf("submit/%s-%d-%s-banner.jpg", community, time.Now().Unix(), user.Login)
	bannerSourceKey := strings.Replace(banner, "https://s3-us-west-2.amazonaws.com/", "", 0)
	err := copyImageToCommunitiesBucket(bannerSourceKey, s3Key)
	if err != nil {
		return nil, err
	}
	return &s3Key, nil
}

func sendEmailNotification(r *http.Request, user twitchapi.UserServiceUser, s3URL string, community string) {
	headers := map[string]string{
		"from":     "Creative Communities <communities@twitch.tv>",
		"to":       "communities@twitch.tv",
		"reply-to": user.Email,
		"subject":  fmt.Sprintf("%s would like to set the banner for %s", user.Login, community),
	}
	client, _ := smtpclient.Client()
	_, err := client.SendMessage(headers, s3URL)
	if err != nil {
		rollbar.RequestError(rollbar.WARN, r, err)
	}
}

// bannerUpload allows users to upload banners
func (s Server) bannerUpload(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	user, err := currentUser(ctx, r)
	if err != nil {
		ServeError(w, r, ErrOauthTokenNotProvided.Error(), http.StatusBadRequest)
		return
	}
	if !user.EmailVerified {
		ServeError(w, r, ErrUserNotVerified.Error(), http.StatusBadRequest)
		return
	}

	file, _, err := r.FormFile("banner")
	if err != nil {
		ServeError(w, r, ErrBannerNotUploaded.Error(), http.StatusBadRequest)
		return
	}
	defer util.CloseAndReport(file)

	if err = validateBanner(r, file); err != nil {
		ServeError(w, r, err.Error(), http.StatusBadRequest)
		return
	}

	s3Key := fmt.Sprintf("preview/%d-%s-banner.jpg", time.Now().Unix(), user.Login)
	err = uploadImageToCommunitiesBucket(s3Key, file)
	if err != nil {
		rollbar.RequestError(rollbar.WARN, r, err)
		ServeError(w, r, ErrBannerNotUploaded.Error(), http.StatusInternalServerError)
		return
	}
	s3URL := s3URL(s3Key)
	resp := BannerUploadResponse{
		BannerURL: s3URL,
	}
	if err := json.NewEncoder(w).Encode(resp); err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
}

// communityProposal will propose a community
func (s Server) communityProposal(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	user, err := currentUser(ctx, r)
	if err != nil {
		ServeError(w, r, ErrOauthTokenNotProvided.Error(), http.StatusBadRequest)
		return
	}
	if !user.EmailVerified {
		ServeError(w, r, ErrUserNotVerified.Error(), http.StatusBadRequest)
		return
	}

	name := r.FormValue("name")
	if name == "" {
		ServeError(w, r, ErrCommunityNotProvided.Error(), http.StatusBadRequest)
		return
	}
	if !models.ValidCommunityName(name) {
		ServeError(w, r, ErrCommunityInvalidName.Error(), http.StatusBadRequest)
		return
	}

	game := r.FormValue("game")
	if !AllowedGames[strings.ToLower(game)] {
		ServeError(w, r, ErrGameNotSupported.Error(), http.StatusNotFound)
		return
	}

	banner := r.FormValue("banner")
	s3Key, err := publishBanner(*user, name, banner)
	if err == ErrPreviewImageNotUploaded {
		ServeError(w, r, ErrPreviewImageNotUploaded.Error(), http.StatusBadRequest)
		return
	}
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
	bannerS3URL := s3URL(*s3Key)

	go sendEmailNotification(r, *user, bannerS3URL, name)

	resp := CommunityProposalResponse{
		Community: CommunityResponse{
			Name:   name,
			Game:   game,
			Banner: bannerS3URL,
		},
	}
	if err := json.NewEncoder(w).Encode(resp); err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
}

// communityList will return a list of communities filtered by user paramaters
func (s Server) communityList(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	game := pat.Param(ctx, "game")
	if !AllowedGames[strings.ToLower(game)] {
		ServeError(w, r, ErrGameNotSupported.Error(), http.StatusNotFound)
		return
	}
	sortBy := r.URL.Query().Get("sortBy")
	if sortBy == "" {
		sortBy = streams
	}
	limit, err := strconv.Atoi(r.URL.Query().Get("limit"))
	if err != nil {
		limit = defaultLimit
	}
	if limit > maxLimit {
		limit = maxLimit
	}
	offset, err := strconv.Atoi(r.URL.Query().Get("offset"))
	if err != nil {
		offset = 0
	}

	communities := []models.Community{}
	if sortBy == frontPage {
		communities, err = getFrontpageCommunities(game, limit)
	} else {
		communities, err = getCommunities(game, sortBy, limit, offset)
	}
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	totalCommunities, err := getCommunitiesCount(game)
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	resp := CommunityListResponse{
		Total:       totalCommunities,
		Communities: convertToCommunityResponses(communities),
		Links:       CommunityListLinks{Self: getSelfURL(r)},
	}
	if err := json.NewEncoder(w).Encode(resp); err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
}

// communityDetail will return a single community by game and name
func (s Server) communityDetail(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	db, err := orm.Client()
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
	game := pat.Param(ctx, "game")
	if !AllowedGames[strings.ToLower(game)] {
		ServeError(w, r, ErrGameNotSupported.Error(), http.StatusNotFound)
		return
	}

	name := pat.Param(ctx, "community")
	if name == "" {
		ServeError(w, r, ErrCommunityNotProvided.Error(), http.StatusBadRequest)
		return
	}
	if !models.ValidCommunityName(name) {
		ServeError(w, r, ErrCommunityInvalidName.Error(), http.StatusBadRequest)
		return
	}

	community, err := models.FindCommunity(game, name, db)
	if err == gorm.ErrRecordNotFound {
		ServeError(w, r, ErrCommunityDoesNotExist.Error(), http.StatusNotFound)
		return
	}
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
	resp := CommunityDetailResponse{
		Community: convertToCommunityResponse(*community),
		Links:     CommunityListLinks{Self: getSelfURL(r)},
	}
	if err := json.NewEncoder(w).Encode(resp); err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
}

// getCommunitiesCount returns the total number of Communities
func getCommunitiesCount(gameName string) (int, error) {
	db, err := orm.Client()
	if err != nil {
		return 0, err
	}
	game, err := models.FindGameByName(gameName, db)
	if err != nil {
		return 0, err
	}
	var count int
	err = db.Model(&models.Community{}).Where("game_id = ?", game.ID).Count(&count).Error
	if err != nil {
		return 0, err
	}
	return count, err
}

// getCommunities returns a list of communities sorted by provided property
func getCommunities(gameName, sortBy string, limit, offset int) ([]models.Community, error) {
	db, err := orm.Client()
	if err != nil {
		return nil, err
	}

	game, err := models.FindGameByName(gameName, db)
	if err != nil {
		return nil, err
	}
	var orderBy string
	switch sortBy {
	case viewers:
		orderBy = "viewer_count DESC"
	case streams:
		orderBy = "stream_count DESC"
	}
	communityList := []models.Community{}
	err = db.Where("game_id = ? AND deleted_at is NULL ", game.ID).Order(orderBy).Limit(limit).Offset(offset).Preload("Game").Find(&communityList).Error
	if err != nil {
		return nil, err
	}
	return communityList, err
}

func getFrontpageCommunities(gameName string, limit int) ([]models.Community, error) {
	viableCommunities, err := getCommunities(gameName, streams, limit*4, 0)
	if err != nil {
		return nil, err
	}
	communities := []models.Community{}
	for i, s := range viableCommunities {
		if i < limit {
			communities = append(communities, s)
		} else {
			if rand.Float64() < 1/float64(i+1) {
				communities[rand.Intn(limit)] = s
			}
		}
	}
	return communities, nil
}

func convertToCommunityResponse(community models.Community) CommunityResponse {
	return CommunityResponse{
		Name:            community.Name,
		Game:            community.Game.Name,
		StreamCount:     community.StreamCount,
		ViewerCount:     community.ViewerCount,
		Banner:          community.Banner(),
		PromotedChannel: community.PromotedChannel,
	}
}

func convertToCommunityResponses(communities []models.Community) []CommunityResponse {
	communityResponses := []CommunityResponse{}
	for _, community := range communities {
		communityResponses = append(communityResponses, convertToCommunityResponse(community))
	}
	return communityResponses
}

// getSelfURL returns the current URL
func getSelfURL(r *http.Request) string {
	u := r.URL
	u.Host = r.Host
	u.Scheme = "http"
	return r.URL.String()
}
