package api

import (
	"encoding/json"
	"net/http"
	"strings"

	"goji.io/pat"

	"code.justin.tv/creative/communities/lib/orm"
	"code.justin.tv/creative/communities/lib/pubsubclient"
	"code.justin.tv/creative/communities/lib/twitchapi"
	"code.justin.tv/creative/communities/models"
	"code.justin.tv/creative/communities/settings"
	"github.com/jinzhu/gorm"
	"github.com/stvp/rollbar"

	"golang.org/x/net/context"
)

const maxFormSize = 32 << 20

// CommunityUpdateRequest is the request for the communityUpdate endpoint
type CommunityUpdateRequest struct {
	User                    twitchapi.UserServiceUser
	Game                    models.Game
	Community               models.Community
	PromotedChannel         *string
	PromotedChannelProvided bool
	Banner                  string
	BannerProvided          bool
}

func communityUpdateParams(ctx context.Context, r *http.Request, db *gorm.DB) (*CommunityUpdateRequest, error) {
	err := r.ParseMultipartForm(maxFormSize)
	if err != nil {
		return nil, ErrRequestInvalid
	}

	user, err := currentUser(ctx, r)
	if err != nil {
		return nil, ErrOauthTokenNotProvided
	}
	if !isWhitelisted(user) {
		return nil, ErrUserNotAuthorized
	}

	gameName := pat.Param(ctx, "game")
	if gameName == "" {
		return nil, ErrGameNotProvided
	}

	communityName := pat.Param(ctx, "community")
	if communityName == "" {
		return nil, ErrCommunityNotProvided
	}

	promotedChannels, promotedChannelProvided := r.Form["promoted_channel"]
	var promotedChannel *string
	if promotedChannelProvided && promotedChannels[0] != "" {
		promotedChannel = &promotedChannels[0]
		var users []twitchapi.UserServiceUser
		users, err = twitchapi.GetUserProperties(ctx, []string{"login"}, nil, []string{*promotedChannel})
		if err != nil || len(users) == 0 {
			return nil, ErrPromotedChannelNotFound
		}
	}

	_, bannerProvided := r.Form["banner"]
	banner := r.FormValue("banner")
	if bannerProvided {
		if banner == "" {
			return nil, ErrBannerNotUploaded
		}
		bannerPrefix := "https://s3-us-west-2.amazonaws.com/" + settings.Resolve("creativeAssetsBucket")
		if !strings.HasPrefix(banner, bannerPrefix) {
			return nil, ErrPreviewImageNotUploaded
		}
	}

	// Validations
	game, err := models.FindGameByName(gameName, db)
	if err != nil {
		if err != models.ErrCommunityNotFound {
			rollbar.RequestError(rollbar.ERR, r, err)
		}
		return nil, ErrGameNotSupported
	}

	community, err := models.FindCommunity(gameName, communityName, db)
	if err != nil {
		if err != models.ErrCommunityNotFound {
			rollbar.RequestError(rollbar.ERR, r, err)
		}
		return nil, ErrCommunityDoesNotExist
	}

	communityUpdateRequest := CommunityUpdateRequest{
		User:                    *user,
		Game:                    *game,
		Community:               *community,
		PromotedChannel:         promotedChannel,
		PromotedChannelProvided: promotedChannelProvided,
		Banner:                  banner,
		BannerProvided:          bannerProvided,
	}
	return &communityUpdateRequest, nil
}

// isWhitelisted returns True when user is whitelisted for privileged endpoints
func isWhitelisted(user *twitchapi.UserServiceUser) bool {
	whitelist := strings.Split(settings.Resolve("whitelist"), ",")
	for _, login := range whitelist {
		if user.Login == login {
			return true
		}
	}
	return false
}

func (s Server) communityUpdate(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	db, err := orm.Client()
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	params, err := communityUpdateParams(ctx, r, db)
	if err != nil {
		ServeError(w, r, err.Error(), http.StatusBadRequest)
		return
	}

	updates := map[string]interface{}{}
	if params.PromotedChannelProvided {
		updates["promoted_channel"] = params.PromotedChannel
	}

	if err = params.Community.UpdateAttributes(db, updates); err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	if params.BannerProvided {
		err := params.Community.PublishBanner(params.Banner)
		if err != nil {
			rollbar.RequestError(rollbar.ERR, r, err)
			ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
			return
		}
	}

	pubsubCtx, _ := context.WithTimeout(ctx, pubsubclient.Timeout)
	sendCommunityUpdateNotifications(pubsubCtx, r, params)

	if err := json.NewEncoder(w).Encode(convertToCommunityResponse(params.Community)); err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
}

func sendCommunityUpdateNotifications(ctx context.Context, r *http.Request, communityUpdateParams *CommunityUpdateRequest) {
	if communityUpdateParams.PromotedChannelProvided {
		if err := pubsubclient.SendPromotedChannelNotification(ctx, &communityUpdateParams.Community); err != nil {
			rollbar.RequestError(rollbar.ERR, r, err)
		}
	}
	if communityUpdateParams.BannerProvided {
		if err := pubsubclient.SendBannerNotification(ctx, &communityUpdateParams.Community); err != nil {
			rollbar.RequestError(rollbar.ERR, r, err)
		}
	}
}
