package api

import (
	"encoding/json"
	"net/http"
	"os"
	"strings"

	"github.com/stvp/rollbar"

	"code.justin.tv/creative/communities/lib/chatcache"
	"code.justin.tv/creative/communities/lib/redisclient"

	"goji.io/pat"

	"golang.org/x/net/context"
)

// getChats gets all the chats for a given community
func (s Server) getChats(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	var hostname, err = os.Hostname()
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	room := "creativecommunity" + pat.Param(ctx, "room")
	redisKey := strings.Join([]string{"chats", hostname, room}, ":")
	redis, err := redisclient.Client()
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	chatJSONs, err := redis.LRange(redisKey, 0, 50).Result()
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
	chats := []chatcache.Chat{}
	for _, chatJSON := range chatJSONs {
		chat := chatcache.Chat{}
		err := json.Unmarshal([]byte(chatJSON), &chat)
		if err != nil {
			rollbar.RequestError(rollbar.ERR, r, err)
			ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
			return
		}
		chats = append(chats, chat)
	}
	chats = reduceBotSpam(chats)
	resp := []string{}
	for _, chat := range chats {
		resp = append(resp, chat.ToIrc())
	}
	if err := json.NewEncoder(w).Encode(&resp); err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
}

func reduceBotSpam(chats []chatcache.Chat) []chatcache.Chat {
	voteAnnouncementCount := 0
	emotesSpammed := map[string]bool{}
	response := []chatcache.Chat{}
	for i := len(chats) - 1; i >= 0; i-- {
		chat := chats[i]
		if chat.Nick == "communitybot" && strings.HasSuffix(chat.Body, " votes") && !strings.HasPrefix(chat.Body, "The vote is finished!") {
			firstWord := strings.Split(chat.Body, " ")[0]
			voteAnnouncementCount++
			if !emotesSpammed[firstWord] {
				emotesSpammed[firstWord] = true
				response = append([]chatcache.Chat{chat}, response...)
			}
		} else {
			response = append([]chatcache.Chat{chat}, response...)
		}
	}

	return response
}
