package api

import (
	"encoding/json"
	"net/http"

	"github.com/stvp/rollbar"

	"code.justin.tv/creative/communities/lib/orm"
	"code.justin.tv/creative/communities/models"

	"golang.org/x/net/context"
)

// GetUserCommunityRequest represents the request for the getUserCommunity endpoint
type GetUserCommunityRequest struct {
	UserGames *[]models.UserGame
}

// UserCommunityResponse represents the response for the getUserCommunity endpoint
type UserCommunityResponse struct {
	UserID        int    `json:"user_id"`
	CommunityName string `json:"community"`
	GameName      string `json:"game"`
}

func getUserCommunityParams(r *http.Request) (*GetUserCommunityRequest, error) {
	// Extract
	userGamesJSON := r.FormValue("user_games")
	if userGamesJSON == "" {
		return nil, ErrUserGamesNotProvided
	}
	userGames := []models.UserGame{}
	err := json.Unmarshal([]byte(userGamesJSON), &userGames)
	if err != nil {
		return nil, ErrUserGamesNotProvided
	}
	getUserCommunityRequest := GetUserCommunityRequest{UserGames: &userGames}
	return &getUserCommunityRequest, nil
}

func convertToUserCommunityResponse(userCommunity *models.UserCommunity) UserCommunityResponse {
	return UserCommunityResponse{
		UserID:        userCommunity.UserID,
		CommunityName: userCommunity.Community.Name,
		GameName:      userCommunity.Game.Name,
	}
}

func convertToUserCommunityResponses(userCommunities *[]models.UserCommunity) []UserCommunityResponse {
	resps := []UserCommunityResponse{}
	for _, userCommunity := range *userCommunities {
		resps = append(resps, convertToUserCommunityResponse(&userCommunity))
	}
	return resps
}

// communityList will return a list of communities filtered by user paramaters
func (s Server) getUserCommunity(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	db, err := orm.Client()
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	params, err := getUserCommunityParams(r)
	if err != nil {
		ServeError(w, r, err.Error(), http.StatusBadRequest)
		return
	}

	userCommunities, err := models.FindUserCommunitiesByUserGame(params.UserGames, db)
	userCommunityResponses := convertToUserCommunityResponses(userCommunities)
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	if err := json.NewEncoder(w).Encode(userCommunityResponses); err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
}
