package api

import (
	"fmt"
	"net/http"
	"sort"
	"strconv"
	"strings"
	"sync"
	"time"

	"github.com/stvp/rollbar"

	"code.justin.tv/creative/communities/lib/redisclient"
	"code.justin.tv/creative/communities/lib/twitchapi"
	"code.justin.tv/creative/communities/models"
	"code.justin.tv/creative/communities/settings"
	"goji.io/pat"
	"golang.org/x/net/context"
)

func cachedResponse(key string, fetcher func() *[]byte, expireIn int) (*[]byte, error) {
	client, err := redisclient.Client()
	if err != nil {
		fmt.Println(err)
		return fetcher(), nil
	}
	fmt.Println("WE CACHE")
	result := client.Get(key)
	if result.Err() != nil {
		result := fetcher()
		cacheValue := string(*result)
		if result != nil {
			fmt.Println(client.Set(key, cacheValue, time.Duration(expireIn)*time.Second))
		}
		return result, nil
	}
	cachedResult, err := result.Bytes()
	return &cachedResult, err
}

// StreamFetcher is used to fetch streams from various sources
type StreamFetcher struct {
	Params  twitchapi.StreamRequestParams
	Fetcher func(twitchapi.StreamRequestParams) (*models.KrakenStreamsResponse, error)
}

// communityAndGameStreams gets both community and game streams and interleave them
func (s Server) communityAndGameStreams(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Cache-Control", "max-age:60, public") // 1 minute cache because this is expensive
	game := r.URL.Query().Get("game")
	if game == "" {
		game = "all"
	}
	if !AllowedGames[strings.ToLower(game)] {
		ServeError(w, r, ErrGameNotSupported.Error(), http.StatusNotFound)
		return
	}

	communityName := pat.Param(ctx, "community")
	if communityName == "" {
		ServeError(w, r, ErrCommunityNotProvided.Error(), http.StatusBadRequest)
		return
	}
	if !models.ValidCommunityName(communityName) {
		ServeError(w, r, ErrCommunityInvalidName.Error(), http.StatusBadRequest)
		return
	}

	limit, err := strconv.Atoi(r.URL.Query().Get("limit"))
	if err != nil {
		limit = 100
	}
	offset, err := strconv.Atoi(r.URL.Query().Get("offset"))
	if err != nil {
		offset = 0
	}
	broadcasterLanguage := r.URL.Query().Get("broadcaster_language")

	fetcher := func() *[]byte {
		var streamFetchwg sync.WaitGroup
		var gameStreams models.KrakenStreams
		var gameErr error
		var communityStreams models.KrakenStreams
		var communityErr error

		streamFetchwg.Add(2)
		go func() {
			defer streamFetchwg.Done()
			gameParams := twitchapi.StreamRequestParams{
				Game:                communityName,
				ClientID:            settings.Resolve("twitchApiClientID"),
				BroadcasterLanguage: broadcasterLanguage,
			}
			gameStreamFetcher := StreamFetcher{
				Params:  gameParams,
				Fetcher: twitchapi.GetStreams,
			}

			gameStreams, gameErr = getAllStreams(gameStreamFetcher)
		}()

		go func() {
			defer streamFetchwg.Done()
			communityParams := twitchapi.StreamRequestParams{
				Game:                "Creative",
				Community:           communityName,
				ClientID:            settings.Resolve("twitchApiClientID"),
				BroadcasterLanguage: broadcasterLanguage,
			}
			communityStreamFetcher := StreamFetcher{
				Params:  communityParams,
				Fetcher: twitchapi.GetCommunityStreams,
			}

			communityStreams, communityErr = getAllStreams(communityStreamFetcher)
		}()

		streamFetchwg.Wait()

		if gameErr != nil {
			rollbar.RequestError(rollbar.ERR, r, err)
			ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
			return nil
		}
		if communityErr != nil {
			rollbar.RequestError(rollbar.ERR, r, err)
			ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
			return nil
		}

		allStreams := append(communityStreams, gameStreams...)
		sort.Sort(allStreams)
		channelNames := []string{}
		for _, s := range allStreams {
			channelNames = append(channelNames, s.Channel.Name)
		}

		params := twitchapi.StreamRequestParams{
			Limit:               limit,
			Offset:              offset,
			Channels:            channelNames,
			BroadcasterLanguage: broadcasterLanguage,
			ClientID:            settings.Resolve("twitchApiClientID"),
		}
		response, err := twitchapi.GetRawStreams(params)
		if err != nil {
			rollbar.RequestError(rollbar.ERR, r, err)
			ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
			return nil
		}
		return response
	}

	key := strings.Join([]string{game,
		communityName,
		r.URL.Query().Get("limit"),
		r.URL.Query().Get("offset"),
		broadcasterLanguage},
		"-")
	response, _ := cachedResponse(key, fetcher, 60)

	fmt.Fprint(w, string(*response))
}

func getAllStreams(p StreamFetcher) (models.KrakenStreams, error) {
	maxStreamsPerPage := 100
	// maxStreamPages, err := cache.GetTotalRequestsRequired(p.Params.Game, maxStreamsPerPage)
	// if err != nil {
	// 	return nil, err
	// }
	maxStreamPages := 5
	var krakenWg sync.WaitGroup
	sigs := make(chan error, 1)
	streamRespsChan := make(chan models.KrakenStream, 1)

	go func() {
		for i := 0; i < maxStreamPages; i++ {
			krakenWg.Add(1)
			go func(offset int) {
				defer krakenWg.Done()
				reqParams := twitchapi.StreamRequestParams{}
				reqParams = p.Params
				reqParams.Limit = 100
				reqParams.Offset = offset
				response, requestErr := p.Fetcher(reqParams)
				if requestErr != nil {
					sigs <- requestErr
				}
				for _, stream := range response.Streams {
					streamRespsChan <- stream
				}
			}(i * maxStreamsPerPage) // current offset
		}
		krakenWg.Wait()
		sigs <- nil
	}()
	streams := models.KrakenStreams{}
getStreamResps:
	for {
		select {
		case s := <-streamRespsChan:
			streams = append(streams, s)
		case err := <-sigs:
			if err != nil {
				return nil, err
			}
			break getStreamResps
		}
	}
	return streams, nil
}
