package api

import (
	"encoding/json"
	"net/http"
	"strconv"

	"code.justin.tv/creative/communities/lib/orm"
	"code.justin.tv/creative/communities/models"
	"github.com/jinzhu/gorm"
	"github.com/stvp/rollbar"

	"golang.org/x/net/context"
)

// SetUserCommunityRequest is the request for the setUserCommunty endpoint
type SetUserCommunityRequest struct {
	UserID    int
	Game      models.Game
	Community models.Community
}

func setUserCommunityParams(r *http.Request, db *gorm.DB) (*SetUserCommunityRequest, error) {
	// Extract
	userID, err := strconv.Atoi(r.FormValue("user_id"))
	if err != nil || userID == 0 {
		return nil, ErrUserNotProvided
	}

	gameName := r.FormValue("game")
	if gameName == "" {
		return nil, ErrGameNotProvided
	}

	communityName := r.FormValue("community")
	if communityName == "" {
		return nil, ErrCommunityNotProvided
	}

	// Validations
	game, err := models.FindGameByName(gameName, db)
	if err != nil {
		return nil, ErrGameNotSupported
	}

	community, err := models.FindCommunity(gameName, communityName, db)
	if err != nil {
		return nil, ErrCommunityDoesNotExist
	}

	userCommunity := SetUserCommunityRequest{
		UserID:    userID,
		Game:      *game,
		Community: *community,
	}
	return &userCommunity, nil
}

// communityList will return a list of communities filtered by user paramaters
func (s Server) setUserCommunity(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	db, err := orm.Client()
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	userCommunityParams, err := setUserCommunityParams(r, db)
	if err != nil {
		ServeError(w, r, err.Error(), http.StatusBadRequest)
		return
	}

	userCommunity := models.UserCommunity{
		GameID:      userCommunityParams.Game.ID,
		CommunityID: userCommunityParams.Community.ID,
		UserID:      userCommunityParams.UserID,
	}

	err = userCommunity.Persist(db)
	resp := convertToUserCommunityResponse(&userCommunity)
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
	if err := json.NewEncoder(w).Encode(resp); err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
}
