package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"
	"strings"

	"github.com/jinzhu/gorm"
	"github.com/stvp/rollbar"

	"code.justin.tv/creative/communities/lib/orm"
	"code.justin.tv/creative/communities/lib/redisclient"
	"code.justin.tv/creative/communities/lib/twitchapi"
	"code.justin.tv/creative/communities/models"
	"code.justin.tv/creative/communities/settings"
	"goji.io/pat"
	"golang.org/x/net/context"
)

func (s Server) communityStreams(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	game := r.URL.Query().Get("game")
	if game == "" {
		game = "all"
	}
	if !AllowedGames[strings.ToLower(game)] {
		ServeError(w, r, ErrGameNotSupported.Error(), http.StatusNotFound)
		return
	}

	communityName := pat.Param(ctx, "community")
	if communityName == "" {
		ServeError(w, r, ErrCommunityNotProvided.Error(), http.StatusBadRequest)
		return
	}
	if !models.ValidCommunityName(communityName) {
		ServeError(w, r, ErrCommunityInvalidName.Error(), http.StatusBadRequest)
		return
	}

	limit, err := strconv.Atoi(r.URL.Query().Get("limit"))
	if err != nil {
		limit = 0
	}
	offset, err := strconv.Atoi(r.URL.Query().Get("offset"))
	if err != nil {
		offset = 0
	}
	broadcasterLanguage := r.URL.Query().Get("broadcaster_language")

	client, err := redisclient.Client()
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
	rediskey := strings.ToLower(strings.Join([]string{"streams", game, communityName}, ":"))
	streamsJSONs, err := client.ZRange(rediskey, int64(0), int64(-1)).Result()
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}
	// Return a 404, but only if the community does not exist. We only check for
	// the community if we get no streams as redis is so much faster.
	if len(streamsJSONs) == 0 {
		db, err := orm.Client()
		if err != nil {
			rollbar.RequestError(rollbar.ERR, r, err)
			ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
			return
		}
		_, err = models.FindCommunity(game, communityName, db)
		if err == gorm.ErrRecordNotFound {
			rollbar.RequestError(rollbar.WARN, r, err)
			ServeError(w, r, ErrCommunityDoesNotExist.Error(), http.StatusNotFound)
			return
		}
		if err != nil {
			rollbar.RequestError(rollbar.ERR, r, err)
			ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
			return
		}
	}

	channelNames := []string{}
	for _, streamJSON := range streamsJSONs {
		s := models.Stream{}
		err := json.Unmarshal([]byte(streamJSON), &s)
		if err != nil {
			field := rollbar.Field{Name: "streamJSON", Data: streamJSON}
			rollbar.Error(rollbar.WARN, err, &field)
		} else {
			channelNames = append(channelNames, s.Name)
		}
	}

	params := twitchapi.StreamRequestParams{
		Game:                game,
		Limit:               limit,
		Offset:              offset,
		Channels:            channelNames,
		BroadcasterLanguage: broadcasterLanguage,
		ClientID:            settings.Resolve("twitchApiClientID"),
	}
	response, err := twitchapi.GetRawStreams(params)
	if err != nil {
		rollbar.RequestError(rollbar.ERR, r, err)
		ServeError(w, r, http.StatusText(http.StatusInternalServerError), http.StatusInternalServerError)
		return
	}

	fmt.Fprint(w, string(*response))
}
