package cache

import (
	"encoding/json"
	"errors"
	"log"
	"math"
	"regexp"
	"strings"
	"sync"

	"github.com/stvp/rollbar"

	"gopkg.in/redis.v3"

	"code.justin.tv/common/config"
	"code.justin.tv/creative/communities/lib/orm"
	"code.justin.tv/creative/communities/lib/redisclient"
	"code.justin.tv/creative/communities/lib/twitchapi"
	"code.justin.tv/creative/communities/models"
	"code.justin.tv/creative/communities/settings"
)

// CommunitiesFromStreams updates local cache of Communities from Streams API
// backs /v1/communities and /v1/streams
func CommunitiesFromStreams() error {
	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}
	maxStreamsPerPage := 100
	// TODO: stop hardcoded call
	maxStreamPages := 10
	if err != nil {
		return err
	}

	var krakenWg sync.WaitGroup
	sigs := make(chan error, 1)
	streamRespsChan := make(chan models.KrakenStream, 1)

	go func() {
		for i := 0; i < maxStreamPages; i++ {
			krakenWg.Add(1)
			go func(offset int) {
				defer krakenWg.Done()
				params := twitchapi.StreamRequestParams{
					Game:     "Creative",
					Limit:    100,
					Offset:   offset,
					ClientID: settings.Resolve("twitchApiClientID"),
				}
				response, requestErr := twitchapi.GetStreams(params)
				if requestErr != nil {
					sigs <- requestErr
				}
				if response != nil {
					for _, stream := range response.Streams {
						streamRespsChan <- stream
					}
				}
			}(i * maxStreamsPerPage) // current offset
		}
		krakenWg.Wait()
		sigs <- nil
	}()

	client, err := redisclient.Client()
	if err != nil {
		return err
	}
	liveCommunitiesByGame := make(map[string]map[string]models.Community)
	liveChannels := make(map[string]bool)

getStreamResps:
	for {
		select {
		case s := <-streamRespsChan:
			communities, extractionErr := extractCommunityNames(s)
			if extractionErr != nil {
				sigs <- extractionErr
			}
			liveChannels[s.Channel.Name] = true
			for _, communityName := range communities {
				for _, game := range []string{s.Game, "all"} {
					key := strings.ToLower(strings.Join([]string{"streams", strings.ToLower(game), communityName}, ":"))
					stream := models.Stream{Name: s.Channel.Name}
					streamJSON, jsonErr := json.Marshal(stream)
					if jsonErr != nil {
						field := rollbar.Field{Name: "json", Data: streamJSON}
						rollbar.Error(rollbar.WARN, errors.New("Invalid Stream JSON"), &field)
					} else {
						val := redis.Z{Score: float64(-s.Viewers), Member: streamJSON}
						client.ZAdd(key, val)
						insensitiveGame := strings.ToLower(game)
						insensitiveCommunity := strings.ToLower(communityName)
						if liveCommunitiesByGame[insensitiveGame] == nil {
							liveCommunitiesByGame[insensitiveGame] = make(map[string]models.Community)
						}
						c := liveCommunitiesByGame[strings.ToLower(game)][insensitiveCommunity]
						c.Game.Name = game
						c.Name = communityName
						c.StreamCount++
						c.ViewerCount += s.Viewers
						liveCommunitiesByGame[insensitiveGame][insensitiveCommunity] = c
					}
				}
			}
		case err = <-sigs:
			if err != nil {
				return err
			}
			break getStreamResps
		}
	}

	if err = updateCommunities(liveChannels, liveCommunitiesByGame); err != nil {
		return err
	}
	if err = purgeInactiveStreams(liveChannels, liveCommunitiesByGame); err != nil {
		return err
	}
	return nil
}

// GetTotalRequestsRequired returns the total number of pages required for request
func GetTotalRequestsRequired(game string, limit int) (int, error) {
	summaryResponse, err := twitchapi.GetStreamSummary("Creative")
	if err != nil {
		return -1, err
	}
	maxStreamPages := int(math.Ceil(float64(summaryResponse.Channels) / float64(limit)))
	return maxStreamPages, nil
}

// extractCommunityNames returns valid Community names from a Stream title
func extractCommunityNames(streamResp models.KrakenStream) ([]string, error) {
	communitySet := make(map[string]bool)
	regex, err := regexp.Compile(`#(\w+)`)
	if err != nil {
		return nil, err
	}
	matches := regex.FindAllStringSubmatch(strings.ToLower(streamResp.Channel.Status), -1)
	for _, match := range matches {
		name := match[1]
		communitySet[name] = true
		if strings.HasSuffix(name, "contest") {
			communitySet[name[:len(name)-7]] = true
		}
	}
	output := []string{}
	for match := range communitySet {
		if !models.BannedCommunities[match] {
			output = append(output, match)
		}
	}
	return output, nil
}

func updateCommunities(liveChannels map[string]bool, liveCommunitiesByGame map[string]map[string]models.Community) error {
	statsd := config.Statsd()
	db, err := orm.Client()
	if err != nil {
		return err
	}

	allCommunities := []models.Community{}
	err = db.Where(true).Preload("Game").Find(&allCommunities).Error
	if err != nil {
		return err
	}

	for _, community := range allCommunities {
		change := liveCommunitiesByGame[strings.ToLower(community.Game.Name)][strings.ToLower(community.Name)]
		attrs := map[string]interface{}{}

		attrs["stream_count"] = change.StreamCount
		key := strings.ToLower(strings.Join([]string{strings.ToLower(community.Game.Name), "stream_count", community.Name}, "."))
		_ = statsd.Gauge(key, int64(change.StreamCount), 1.0)

		attrs["viewer_count"] = change.ViewerCount
		key = strings.ToLower(strings.Join([]string{strings.ToLower(community.Game.Name), "viewer_count", community.Name}, "."))
		_ = statsd.Gauge(key, int64(change.ViewerCount), 1.0)

		if community.PromotedChannel != nil && !liveChannels[*community.PromotedChannel] {
			attrs["promoted_channel"] = nil
		}
		err := community.UpdateAttributes(db, attrs)
		if err != nil {
			rollbar.Error(rollbar.WARN, err)
		}
	}
	return nil
}

// purgeInactiveStreams removes all channels that are not live from cache
func purgeInactiveStreams(liveChannels map[string]bool, liveCommunitiesByGame map[string]map[string]models.Community) error {
	client, err := redisclient.Client()
	if err != nil {
		return err
	}
	allCommunities, err := client.Keys("streams:*").Result()
	if err != nil {
		return err
	}
	for _, cKey := range allCommunities {
		splitKey := strings.Split(cKey, ":")
		if len(splitKey) == 3 {
			game := splitKey[1]
			communityName := splitKey[2]
			if liveCommunitiesByGame[game][communityName] != (models.Community{}) {
				allStreamsJSON, err := client.ZRange(cKey, 0, -1).Result()
				if err != nil {
					return err
				}
				inactiveChannels := []string{}
				for _, streamJSON := range allStreamsJSON {
					s := models.Stream{}
					err := json.Unmarshal([]byte(streamJSON), &s)
					if err != nil {
						field := rollbar.Field{Name: "json", Data: streamJSON}
						rollbar.Error(rollbar.WARN, errors.New("Invalid Stream JSON"), &field)
					} else if !liveChannels[s.Name] {
						inactiveChannels = append(inactiveChannels, streamJSON)
					}
				}
				// Delete offline channels
				if len(inactiveChannels) > 0 {
					client.ZRem(cKey, inactiveChannels...)
				}
			} else {
				// No streams left within this community, delete entirely
				client.Del(cKey)
			}
		}
	}
	return nil
}
