package chatcache

import (
	"encoding/json"
	"fmt"
	"os"
	"regexp"
	"strings"
	"time"

	"github.com/stvp/rollbar"
	"gopkg.in/redis.v3"

	"code.justin.tv/chat/tmi/sse"
	"code.justin.tv/creative/communities/lib/redisclient"
)

// Chat is a chat message
type Chat struct {
	Command   string
	Room      string
	Nick      string
	Body      string
	Tags      string
	Timestamp int64
	Target    string `json:",omitempty"`
}

// Run runs the chat cacher
func Run() error {
	hostname, err := os.Hostname()
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}

	redisClient, err := redisclient.Client()
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	firehoseSource, err := sse.NewEventSource("http://tmi.twitch.tv/firehose")
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}
	err = firehoseSource.Connect()
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}

	errChan := make(chan error, 1)
	chatChan := make(chan Chat, 100)
	go func() {
		for {
			event, firehoseErr := firehoseSource.Get()
			if firehoseErr != nil {
				errChan <- firehoseErr
				continue
			}
			c, err := parseChat(event)
			if err != nil {
				rollbar.Error(rollbar.ERR, err)
				continue
			}
			chatChan <- c
		}
	}()

	for {
		select {
		case chat := <-chatChan:
			room := chat.Room
			if room == "" {
				room = chat.Target
			}
			if strings.HasPrefix(room, "creativecommunity") {
				if chat.Command == "CLEARCHAT" {
					if chat.Body == "" {
						err := clearChatCache(redisClient, hostname, room)
						if err != nil {
							rollbar.Error(rollbar.ERR, err)
						}
					} else {
						err := removeMessagesByUser(redisClient, hostname, room, chat.Body)
						if err != nil {
							rollbar.Error(rollbar.ERR, err)
							// Clear whole chat cache just in calse
							err := clearChatCache(redisClient, hostname, room)
							if err != nil {
								rollbar.Error(rollbar.ERR, err)
							}
						}
					}
				} else {
					err := saveChatMessage(redisClient, hostname, room, chat)
					if err != nil {
						rollbar.Error(rollbar.ERR, err)
					}
				}
			}
		case err := <-errChan:
			return err
		}

	}
}

func saveChatMessage(redisClient *redis.Client, hostname string, room string, chat Chat) error {
	jsonChat, err := json.Marshal(chat)

	if err != nil {
		return err
	}
	redisClient.RPush(redisKey(hostname, room), string(jsonChat))
	redisClient.LTrim(redisKey(hostname, room), -50, -1)
	return nil
}

func clearChatCache(redisClient *redis.Client, hostname string, room string) error {
	return redisClient.Del(redisKey(hostname, room)).Err()
}

func removeMessagesByUser(redisClient *redis.Client, hostname string, room string, login string) error {
	messages, redisErr := redisClient.LRange(redisKey(hostname, room), 0, 50).Result()
	if redisErr != nil {
		return redisErr
	}
	for _, messageJSON := range messages {
		var message Chat
		if err := json.Unmarshal([]byte(messageJSON), &message); err != nil {
			return err
		}
		if login == message.Nick {
			_, err := redisClient.LRem(redisKey(hostname, room), 0, messageJSON).Result()
			if err != nil {
				return err
			}
		}
	}
	return nil
}

func redisKey(hostname string, room string) string {
	return strings.Join([]string{"chats", hostname, room}, ":")
}

func removeHash(str string) string {
	if strings.HasPrefix(str, "#") {
		return str[1:]
	}
	return str
}

// Copied from https://git-aws.internal.justin.tv/video/rechat_ingester/blob/0d2e628ab059f254530e1e779eccfa8b57c5439e/worker.go
func parseChat(event *sse.Event) (Chat, error) {
	var chat Chat
	err := json.Unmarshal([]byte(event.Data), &chat)
	if err != nil {

		rollbar.Error(rollbar.ERR, err)

		return Chat{}, err
	}
	chat.Room = removeHash(chat.Room)
	chat.Target = removeHash(chat.Target)

	//Handle /me chats
	re := regexp.MustCompile("^\u0001ACTION ([^\u0001]+)\u0001$")
	match := re.FindStringSubmatch(chat.Body)
	if match != nil {
		chat.Body = match[1]
		chat.Tags += ";style=action;"
	}

	currentTime := time.Now()
	chat.Timestamp = currentTime.UnixNano() / (int64(time.Millisecond) / int64(time.Nanosecond))
	return chat, nil
}

// ToIrc formats the message for IRC
func (c Chat) ToIrc() string {
	// "@badges=staff/1,turbo/1;color=#00AACC;display-name=jimmyhillis;emotes=;id=5aab4546-1eb7-4600-bfb7-df79e19978f3;mod=0;room-id=64366426;subscriber=0;turbo=1;user-id=100728974;user-type=staff :jimmyhillis!jimmyhillis@jimmyhillis.tmi.twitch.tv PRIVMSG #jwahba :hey testing yeah"
	return fmt.Sprintf("@%s :%s!%s@%s.tmi.twitch.tv PRIVMSG #%s :%s", c.Tags, c.Nick, c.Nick, c.Nick, c.Room, c.Body)
}
