package pubsubclient

import (
	"encoding/json"
	"fmt"
	"time"

	pubclient "code.justin.tv/chat/pubsub-go-pubclient/client"
	"code.justin.tv/common/config"
	"code.justin.tv/creative/communities/models"
	"code.justin.tv/creative/communities/settings"
	"golang.org/x/net/context"
)

const (
	topicPrefix = "creative-community"
	// Timeout expected to add to context for pubsub wait
	Timeout = 2 * time.Second
)

func pubsubCommunityTopic(community *models.Community) string {
	return fmt.Sprintf("%s.%s", topicPrefix, community.Name)
}

type pubsubPromotedChannelMessage struct {
	Type            string  `json:"type"`
	PromotedChannel *string `json:"promoted_channel"`
}

// SendPromotedChannelNotification notifies clients when a Community Promoted Channel has changed
func SendPromotedChannelNotification(ctx context.Context, community *models.Community) error {
	topic := pubsubCommunityTopic(community)
	message, err := json.Marshal(pubsubPromotedChannelMessage{
		Type:            "promoted-channel",
		PromotedChannel: community.PromotedChannel,
	})
	if err != nil {
		return err
	}
	return SendPubsubMessage(ctx, topic, string(message))
}

type pubsubBannerMessage struct {
	Type   string `json:"type"`
	Banner string `json:"banner"`
}

// SendBannerNotification notifies clients when a Community banner has changed
func SendBannerNotification(ctx context.Context, community *models.Community) error {
	topic := pubsubCommunityTopic(community)
	message, err := json.Marshal(pubsubBannerMessage{
		Type:   "banner",
		Banner: community.Banner(),
	})
	if err != nil {
		return err
	}
	return SendPubsubMessage(ctx, topic, string(message))
}

// SendPubsubMessage will push a notification to the pubsub broker
func SendPubsubMessage(ctx context.Context, topic, message string) error {
	stats := config.Statsd()
	client, err := pubclient.NewPubClient(pubclient.PubConfig{
		Host:        settings.Resolve("pubsubBroker"),
		Stats:       stats,
		StatsPrefix: "communities.pubsub_client",
	})
	if err != nil {
		return err
	}
	return client.Publish(ctx, []string{topic}, message, nil)
}
