package twitchapi

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/url"
	"strconv"
	"strings"

	"github.com/stvp/rollbar"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/creative/communities/lib"
	"code.justin.tv/creative/communities/models"
	"code.justin.tv/creative/communities/settings"
	"golang.org/x/net/context"
)

const apiDomain = "https://api.twitch.tv"

// request will return the response from any Twitch API request
func request(path string) (*[]byte, error) {
	client, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host: apiDomain,
	})
	if err != nil {
		return nil, err
	}
	req, err := client.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}
	resp, err := client.Do(context.Background(), req, twitchhttp.ReqOpts{
		StatName: "communities_update_redis",
	})
	if err != nil {
		return nil, err
	}
	defer util.CloseAndReport(resp.Body)
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	return &body, nil
}

// GetStreamSummary returns a summary of streams for a specific game
func GetStreamSummary(game string) (*models.KrakenStreamsSummaryResponse, error) {
	params := make(url.Values)
	params.Set("game", game)
	params.Set("client_id", settings.Resolve("twitchApiClientID"))
	path := fmt.Sprintf("https://api.twitch.tv/kraken/streams/summary?%s", params.Encode())
	body, err := request(path)
	if err != nil {
		return nil, err
	}
	resp := models.KrakenStreamsSummaryResponse{}
	err = json.Unmarshal(*body, &resp)
	if err != nil {
		return nil, err
	}
	return &resp, nil
}

// StreamRequestParams can be provided to GetStreams to filter
// the streams returned
type StreamRequestParams struct {
	Game                string
	Limit               int
	Offset              int
	Channels            []string
	ClientID            string
	StreamType          string
	BroadcasterLanguage string
	Community           string
}

// GetRawStreams returns raw []byte response from provided filters
func GetRawStreams(p StreamRequestParams) (*[]byte, error) {
	params := make(url.Values)
	if p.Game != "" {
		params.Set("game", p.Game)
	}
	if p.Limit != 0 {
		params.Set("limit", strconv.Itoa(p.Limit))
	}
	if p.Offset != 0 {
		params.Set("offset", strconv.Itoa(p.Offset))
	}
	if p.Channels != nil {
		params.Set("channel", strings.Join(p.Channels, ","))
	}
	if p.ClientID != "" {
		params.Set("client_id", p.ClientID)
	}
	if p.StreamType != "" {
		params.Set("stream_type", p.StreamType)
	}
	if p.BroadcasterLanguage != "" {
		params.Set("broadcaster_language", p.BroadcasterLanguage)
	}
	path := fmt.Sprintf("/kraken/streams?%s", params.Encode())
	return request(path)
}

// GetStreams returns []models.krakenStream from provided filters
func GetStreams(p StreamRequestParams) (*models.KrakenStreamsResponse, error) {
	body, err := GetRawStreams(p)
	if err != nil {
		return nil, err
	}
	resp := models.KrakenStreamsResponse{}
	err = json.Unmarshal(*body, &resp)
	if err != nil {
		return nil, err
	}
	// JSON response from Streams API is in PST/PDT but has no timezone
	// associated with the response (this manually applies the timezone)
	// without changing the time itself
	for i, stream := range resp.Streams {
		createdAtInPst, err := util.SetPst(*stream.CreatedAt)
		if err != nil {
			field := rollbar.Field{Name: "dateString", Data: stream.CreatedAt}
			rollbar.Error(rollbar.WARN, errors.New("Invalid Created At date"), &field)
		}
		resp.Streams[i].CreatedAt = createdAtInPst
	}
	return &resp, nil
}

// GetRawCommunityStreams returns raw []byte response from provided filters
func GetRawCommunityStreams(p StreamRequestParams) (*[]byte, error) {
	params := make(url.Values)
	if p.Game != "" {
		params.Set("game", p.Game)
	}
	if p.Limit != 0 {
		params.Set("limit", strconv.Itoa(p.Limit))
	}
	if p.Offset != 0 {
		params.Set("offset", strconv.Itoa(p.Offset))
	}
	if p.Channels != nil {
		params.Set("channel", strings.Join(p.Channels, ","))
	}
	if p.ClientID != "" {
		params.Set("client_id", p.ClientID)
	}
	if p.StreamType != "" {
		params.Set("stream_type", p.StreamType)
	}
	if p.BroadcasterLanguage != "" {
		params.Set("broadcaster_language", p.BroadcasterLanguage)
	}
	path := fmt.Sprintf("/kraken/streams/communities/%s?%s", p.Community, params.Encode())
	return request(path)
}

// GetCommunityStreams returns []models.krakenStream from provided filters
func GetCommunityStreams(p StreamRequestParams) (*models.KrakenStreamsResponse, error) {
	body, err := GetRawCommunityStreams(p)
	if err != nil {
		return nil, err
	}
	resp := models.KrakenStreamsResponse{}
	err = json.Unmarshal(*body, &resp)
	if err != nil {
		return nil, err
	}
	// JSON response from Streams API is in PST/PDT but has no timezone
	// associated with the response (this manually applies the timezone)
	// without changing the time itself
	for i, stream := range resp.Streams {
		createdAtInPst, err := util.SetPst(*stream.CreatedAt)
		if err != nil {
			field := rollbar.Field{Name: "dateString", Data: stream.CreatedAt}
			rollbar.Error(rollbar.WARN, errors.New("Invalid Created At date"), &field)
		}
		resp.Streams[i].CreatedAt = createdAtInPst
	}
	return &resp, nil
}
