package twitchapi

import (
	"encoding/json"
	"errors"
	"io/ioutil"
	"net/url"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/creative/communities/lib"
	"code.justin.tv/creative/communities/settings"
	"github.com/stvp/rollbar"
)

// UserServiceResponse is the container for the objects returned from the user service
type UserServiceResponse struct {
	Result []UserServiceUser `json:"results"`
}

// UserServiceUser is the user object returned from the user service
type UserServiceUser struct {
	ID            int    `json:"id"`
	Email         string `json:"email"`
	Login         string `json:"login"`
	EmailVerified bool   `json:"email_verified"`
}

// ErrUnverifiedUser is generated whent the currentUser could not be found
var ErrUnverifiedUser = errors.New("Could not verify user id")

// ErrNoIdsOrLoginsProvided is generated when neither the ids nor logins are provided
var ErrNoIdsOrLoginsProvided = errors.New("No id or login provided")

// GetUserProperties Gets all user properties specified by commaseparated ids or logins
func GetUserProperties(ctx context.Context, properties, ids, logins []string) ([]UserServiceUser, error) {

	if len(logins) == 0 && len(ids) == 0 {
		return nil, ErrNoIdsOrLoginsProvided
	}

	v := url.Values{}
	for _, login := range logins {
		v.Add("login", login)
	}

	for _, id := range ids {
		v.Add("id", string(id))
	}

	for _, prop := range properties {
		v.Add("prop", prop)
	}

	client, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host: settings.Resolve("userServiceURL"),
	})
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		return nil, ErrUnverifiedUser
	}

	req, err := client.NewRequest("GET", "/users?"+v.Encode(), nil)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		return nil, ErrUnverifiedUser
	}

	resp, err := client.Do(ctx, req, twitchhttp.ReqOpts{
		StatName: "communities_get_user_properties",
	})
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		return nil, ErrUnverifiedUser
	}

	defer util.CloseAndReport(resp.Body)

	userServiceResponse := UserServiceResponse{}
	err = json.Unmarshal(body, &userServiceResponse)
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		return nil, ErrUnverifiedUser
	}
	return userServiceResponse.Result, nil
}
