package models

import (
	"fmt"
	"regexp"
	"strings"
	"time"

	"code.justin.tv/creative/communities/lib/s3client"
	"code.justin.tv/creative/communities/settings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/jinzhu/gorm"
	"github.com/stvp/rollbar"
)

// Community represents a community
type Community struct {
	ID              int        `json:"-"`
	Name            string     `json:"name"`
	Game            Game       `json:"game"`
	PromotedChannel *string    `json:"promoted_channel"`
	GameID          int        `json:"-"`
	StreamCount     int        `json:"-"`
	ViewerCount     int        `json:"-"`
	UpdatedAt       time.Time  `json:"-"`
	CreatedAt       time.Time  `json:"-"`
	DeletedAt       *time.Time `json:"-"`
}

// Banner is where the community's banner url is stored
func (c Community) Banner() string {
	return fmt.Sprintf("https://s3-us-west-2.amazonaws.com/%s/%s", settings.Resolve("creativeAssetsBucket"), c.BannerPath())
}

// BannerPath is the path to the banner within the bucket
func (c Community) BannerPath() string {
	return fmt.Sprintf("communities/%s.jpg", c.Name)
}

// PublishBanner sets the banner for a given community
func (c Community) PublishBanner(bannerURL string) error {
	s3SourceKey := strings.Replace(bannerURL, "https://s3-us-west-2.amazonaws.com/", "", 0)
	assetsClient := s3client.Client()
	_, err := assetsClient.CopyObject(&s3.CopyObjectInput{
		Bucket:     aws.String(settings.Resolve("creativeAssetsBucket")),
		Key:        aws.String(c.BannerPath()),
		CopySource: aws.String(s3SourceKey),
		ACL:        aws.String("public-read"),
	})
	return err
}

// BannedCommunities is a list of banned communities
var BannedCommunities = map[string]bool{
	"retarded":        true,
	"hijoeputa":       true,
	"pajera":          true,
	"putito":          true,
	"wetback":         true,
	"serefsiz":        true,
	"culeado":         true,
	"estupidos":       true,
	"cojudo":          true,
	"jackass":         true,
	"fuckyou":         true,
	"idiotas":         true,
	"maldito":         true,
	"conchetumadre":   true,
	"babosas":         true,
	"conchetumare":    true,
	"malparidas":      true,
	"prostituta":      true,
	"babosos":         true,
	"cabronas":        true,
	"chuchon":         true,
	"conchasumadres":  true,
	"pastrula":        true,
	"pastrulas":       true,
	"encule":          true,
	"pastrulo":        true,
	"pastrulos":       true,
	"cagados":         true,
	"gilipoya":        true,
	"reconchatumadre": true,
	"chuchas":         true,
	"culona":          true,
	"cerote":          true,
	"malparidos":      true,
	"conchasumadre":   true,
	"fagget":          true,
	"babosa":          true,
	"gilipolla":       true,
	"chinguen":        true,
	"bitches":         true,
	"sudaca":          true,
	"indios":          true,
	"malparida":       true,
	"chingado":        true,
	"bullshit":        true,
	"enculer":         true,
	"mames":           true,
	"dickhead":        true,
	"gilipoyas":       true,
	"zorras":          true,
	"bicha":           true,
	"cagar":           true,
	"marika":          true,
	"mamon":           true,
	"cocksucker":      true,
	"culiado":         true,
	"pendeja":         true,
	"vadia":           true,
	"culeros":         true,
	"gonorrea":        true,
	"mamar":           true,
	"maricas":         true,
	"vergas":          true,
	"mierdas":         true,
	"estupida":        true,
	"huevon":          true,
	"atdhe":           true,
	"pajero":          true,
	"sikeyim":         true,
	"skank":           true,
	"cagada":          true,
	"amcik":           true,
	"berga":           true,
	"perro":           true,
	"conchatumadre":   true,
	"reputas":         true,
	"reputo":          true,
	"reputos":         true,
	"reputa":          true,
	"merde":           true,
	"siktir":          true,
	"carajo":          true,
	"cazzo":           true,
	"trola":           true,
	"baboso":          true,
	"fucked":          true,
	"connard":         true,
	"fucking":         true,
	"pinga":           true,
	"gilipollas":      true,
	"imbecil":         true,
	"cagadas":         true,
	"salope":          true,
	"cabros":          true,
	"polla":           true,
	"yarrak":          true,
	"pendejos":        true,
	"retard":          true,
	"cagones":         true,
	"indio":           true,
	"culiao":          true,
	"cabro":           true,
	"wanker":          true,
	"niggers":         true,
	"pelotudo":        true,
	"queer":           true,
	"sikerim":         true,
	"indigena":        true,
	"boludo":          true,
	"chucha":          true,
	"homosexual":      true,
	"zorra":           true,
	"pinche":          true,
	"hijueputa":       true,
	"sexy":            true,
	"orospu":          true,
	"putas":           true,
	"malparido":       true,
	"porn":            true,
	"porno":           true,
	"cabrones":        true,
	"culero":          true,
	"putos":           true,
	"concha":          true,
	"buceta":          true,
	"bastard":         true,
	"chinga":          true,
	"viado":           true,
	"marica":          true,
	"motherfucker":    true,
	"boob":            true,
	"merda":           true,
	"caralho":         true,
	"cabron":          true,
	"suck":            true,
	"maricon":         true,
	"faggot":          true,
	"vagina":          true,
	"nigga":           true,
	"penis":           true,
	"pendejo":         true,
	"fuck":            true,
	"verga":           true,
	"asshole":         true,
	"mierda":          true,
	"pussy":           true,
	"nigger":          true,
	"whore":           true,
	"slut":            true,
	"fag":             true,
	"sex":             true,
	"dickcunt":        true,
	"shit":            true,
	"bitch":           true,
	"hore":            true,
	"pinto":           true,
	"rape":            true,
	"twat":            true,
}

// ValidCommunityName returns True if provided name is valid
// A valid community name is 2-21 alphanumeric characters (allowing _)
// and is not a banned name.
func ValidCommunityName(communityName string) bool {
	matchesRegex, err := regexp.MatchString("^[A-Za-z0-9][A-Za-z0-9_]{2,30}$", communityName)
	return !BannedCommunities[communityName] && matchesRegex && err == nil
}

// FindCommunity finds a community by it's case insensitive name
func FindCommunity(gameName, communityName string, db *gorm.DB) (*Community, error) {
	game, err := FindGameByName(gameName, db)
	if err != nil {
		return nil, err
	}
	community := Community{}
	err = db.Where("lower(name) = lower(?) AND game_id = ? AND deleted_at IS NULL", communityName, game.ID).First(&community).Error
	if err != nil {
		if err != gorm.ErrRecordNotFound {
			return nil, ErrCommunityNotFound
		}
		return nil, err
	}
	community.Game = *game
	return &community, nil
}

// UpdateAttributes saves the community using first or create
func (c *Community) UpdateAttributes(db *gorm.DB, attrs map[string]interface{}) error {
	where := Community{
		ID:     c.ID,
		GameID: c.GameID,
	}

	err := db.Where(where).First(&c).Error
	if err != nil {
		rollbar.Error(rollbar.ERR, ErrDBSaveFailed)
		return ErrDBSaveFailed
	}

	err = db.Model(&c).Update(attrs).Error
	if err != nil {
		rollbar.Error(rollbar.ERR, ErrDBSaveFailed)
		return ErrDBSaveFailed
	}
	return nil
}
