package models

import (
	"time"

	"github.com/jinzhu/gorm"
	"github.com/stvp/rollbar"
)

// UserCommunity contains the relationship between a user and the community
type UserCommunity struct {
	ID          int
	UserID      int
	GameID      int
	CommunityID int
	Community   Community
	Game        Game
	UpdatedAt   time.Time
	CreatedAt   time.Time
}

// Persist saves the user community using first or create
func (uc *UserCommunity) Persist(db *gorm.DB) error {
	where := UserCommunity{
		UserID: uc.UserID,
		GameID: uc.GameID,
	}
	attrs := UserCommunity{CommunityID: uc.CommunityID}

	err := db.Where(where).Assign(attrs).Preload("Game").Preload("Community").FirstOrCreate(&uc).Error
	if err != nil {
		rollbar.Error(rollbar.ERR, ErrDBSaveFailed)
		return ErrDBSaveFailed
	}
	return nil
}

// UserGame is used to extract a usergame pair
type UserGame struct {
	GameID int `json:"game_id"`
	UserID int `json:"user_id"`
}

// FindUserCommunitiesByUserGame finds a user community by userids
func FindUserCommunitiesByUserGame(userGames *[]UserGame, db *gorm.DB) (*[]UserCommunity, error) {
	userCommunities := []UserCommunity{}
	query := db
	for _, userGame := range *userGames {
		if userGame.GameID != 0 && userGame.UserID != 0 {
			where := UserCommunity{
				GameID: userGame.GameID,
				UserID: userGame.UserID,
			}
			query = query.Or(where)
		}
	}
	err := query.Preload("Game").Preload("Community").Find(&userCommunities).Error
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
		return nil, err
	}
	return &userCommunities, nil
}
