import datetime
import sys
from communitybot.community import (
    Community,
    VoteAlreadyInProgressError,
    NoLiveChannels
)
import communitybot.logger as logger
from communitybot.settings import SETTINGS
from communitybot.stats import gauge
from communitybot.clients.twitch import TwitchApi, TwitchApiError
from communitybot.bots.twitch import TwitchBot
from communitybot.util import (
    command_throttle,
    has_matches,
    BAN_WORDS,
    TIMEOUT_WORDS
)

DEFAULT_VOTE_DURATION = datetime.timedelta(minutes=2)
DEFAULT_TIMEOUT = datetime.timedelta(seconds=10)


class CommunityBot(TwitchBot):
    '''
    Community Bot supports moderating and controlling the Community
    Theatre for any number of Communities at the same time.
    '''

    COMMANDS = [
        '!help',
        '!removeme',
        '!whatsnext'
    ]
    MOD_COMMANDS = [
        '!request',
        '!settopic',
        '!tellbroadcaster'
    ]
    ADMIN_COMMANDS = [
        '!announce',
        '!killrequest',
        '!removebroadcaster',
        '!requestover',
        '!requesttime',
        '!requesttimeover',
        '!setbroadcaster'
    ]

    COMMAND_DESCRIPTIONS = {
        '!help': '"!help <command>" to see this helpful message',
        '!removeme': 'Remove yourself from this Community\'s Spotlight',
        '!request': 'Change the channel by running a user vote',
        '!settopic': '"!settopic <title>" to change this community\'s topic',
        '!tellbroadcaster': ('"!tellbroadcater <message>" to send a message '
                             'to the broadcaster in the Spotlight'),
        '!whatsnext': 'Find out what we\'re doing next'
    }

    def __init__(self, **kwargs):
        communities = [Community(**c) for c in kwargs.pop('communities')]
        channels = [c.channel for c in communities]
        super(CommunityBot, self).__init__(channels=channels, **kwargs)
        self.communities = dict(zip(channels, communities))

    def set_mods(self, channel, mods=[]):
        '''
        Track each moderator within a community room and ensure the
        bot is a moderator, otherwise speak loudly
        '''
        super(CommunityBot, self).set_mods(channel, mods)
        community = self.communities[channel]
        community.mods = mods
        if community.bot_is_moderator is False:
            logger.log("%s is not a moderator on %s" %
                       (SETTINGS['nickname'], community.name),
                       logger.ERROR)

    def on_pubmsg(self, conn, event):
        '''
        Track each message and parse for commands
        '''
        try:
            user, channel, msg = self.get_user_msg_from_irc(event)
            community = self.communities[channel]
            # if self.should_ban(msg):
            #     return self.ban(conn, user, community.channel)
            # if self.should_timeout(msg):
            #     return self.timeout(conn, user, community.channel,
            #                         DEFAULT_TIMEOUT)
            commands = self.COMMANDS[:]
            if community.is_mod(user):
                commands += self.MOD_COMMANDS[:]
            if community.is_admin(user):
                commands += self.ADMIN_COMMANDS[:]
            command, args = self.parse_command(msg, commands)
            if command:
                return self.do_command(conn, user, community, command, args)
            if (community.vote is not None and
                    community.vote.is_valid_vote(msg)):
                return community.vote.vote(user, msg)
        except Exception as e:
            logger.exception(sys.exc_info())

    def on_join(self, conn, event):
        super(CommunityBot, self).on_join(conn, event)
        community = self.communities[event.target]
        if community.is_tracking is False:
            community.is_tracking = True
            args = (conn, community)
            self.update_promoted_channel(conn, community)
            self.reactor.execute_every(5, self.update_promoted_channel, args)
            self.reactor.execute_every(SETTINGS['chat_information_frequency'],
                                       self.tell_chat_whats_next, args)
            self.reactor.execute_every(1, self.handle_request_times, args)
            self.reactor.execute_every(30, self.update_dashboard, args)
            if SETTINGS['requesttime_on_boot'] is True:
                self.request_time(conn, community, None)

    def should_ban(self, msg):
        '''
        Return True when message should cause the the user to be banned
        '''
        return has_matches(msg, BAN_WORDS)

    def should_timeout(self, msg):
        '''
        Return True when message should cause the the user to have a timeout
        '''
        return has_matches(msg, TIMEOUT_WORDS)

    def send_community_pubsub(self, community, message):
        '''
        Send a pubsub message to the provided community
        '''
        topic = 'creative-community.%s' % community.name
        self.send_pubsub(topic, message)

    def do_command(self, conn, user, community, command, args=()):
        '''
        Run the provided command against the provided connection
        and channel
        '''
        is_admin = community.is_admin(user)
        if command == '!settopic':
            self.set_topic(conn, community, user, args)
        elif command == '!requesttime':
            self.request_time(conn, community, user)
        elif command == '!announce':
            self.announce(conn, community, user, *args)
        elif command == '!tellbroadcaster':
            self.tell_broadcaster(conn, community, user, *args)
        elif command == '!requesttimeover':
            self.request_time_over(conn, community, user)
        elif command == '!request' and len(args) <= 2:
            self.request(conn, community, user, *args)
        elif command == '!requestover':
            self.request_over(conn, community, user)
        elif command == '!killrequest':
            self.request_over(conn, community, user, ignore_results=True)
        elif command == '!setbroadcaster' and len(args) == 1:
            self.set_broadcaster(conn, community, user, args[0])
        elif command == '!removeme':
            self.remove_broadcaster(conn, community, user)
        elif command == '!removebroadcaster':
            self.remove_broadcaster(conn, community, user, validate_user=False)
        elif command == '!whatsnext':
            self.whats_next(conn, community, user)
        elif command == '!help':
            command = None
            if args:
                command = args.pop()
                if command[0] != '!':
                    command = '!%s' % command
            self.help(conn, community, user, command=command)

    def proxy(self, conn, channel, message, community):
        '''
        Proxy message from the current channel to the provided channel
        '''
        if SETTINGS['debug']:
            self.say(conn, community.channel,
                     'PROXY %s: %s' % (channel, message))
            return
        self.say(conn, channel, message)

    @command_throttle(seconds=10)
    def help(self, conn, community, user, command=None):
        '''
        Send help message for each command
        '''
        channel = community.channel
        if command is not None:
            try:
                self.say(conn, channel, self.COMMAND_DESCRIPTIONS[command])
            except KeyError:
                self.say(conn, channel, 'I\'m afraid I can\'t do that')
            return
        users_commands = self.COMMANDS[:]
        if community.is_mod(user):
            users_commands += self.MOD_COMMANDS[:]
        command_list = ', '.join(users_commands)
        self.say(conn, channel, 'Yo %s you can do %s' % (user, command_list))
        self.say(conn, channel, 'If you need help with something specific you '
                 'can do `!help <command>`')

    @command_throttle(seconds=10)
    def whats_next(self, conn, community, user):
        '''
        Send a message to list the next time a vote will happen
        '''
        if community.vote is not None:
            self.say(conn, community.channel, 'We\'re voting right now! You '
                     'are in control of your own destiny. Kind of.')
            return
        next_vote = self.next_vote(community)
        if community.is_request_time is False or next_vote is None:
            self.say(conn, community.channel, 'No plans. What you wanna do?')
            return
        delta = next_vote - datetime.datetime.now()
        minutes = delta.seconds // 60
        if minutes < 1:
            self.say(conn, community.channel, 'We\'re voting any second!')
            return
        message = ('We\'ll change channels in about %d minutes - '
                   'get ready to vote!' % minutes)
        self.say(conn, community.channel, message)

    def announce(self, conn, community, user, *args):
        '''
        Send a message to all communities
        '''
        message = ' '.join(args)
        for channel in [c.channel for c in self.communities.values()]:
            self.proxy(conn, channel, message, community)

    def tell_broadcaster(self, conn, community, user, *args):
        '''
        Send a message to broadcaster channel
        '''
        if not community.promoted_channel:
            self.say(conn, community.channel,
                     'They aren\'t answering. Maybe they aren\'t home?')
            return
        channel = '#%s' % community.promoted_channel
        message = '%s says %s from %s community' % (
            user, ' '.join(args), community.name)
        self.proxy(conn, channel, message, community)

    def set_broadcaster(self, conn, community, user, channel):
        '''
        Update the promoted channel for a community and notify
        all parties of the change
        '''
        try:
            TwitchApi.set_promoted_channel(community.name, channel)
            self.update_promoted_channel(conn, community)
            if channel is not None:
                community_url = "%s/directory/game/Creative/%s" % (
                    SETTINGS['twitch_web'], community.name)
                rerun = None
                if community.vote:
                    rerun = community.vote.rerun
                if rerun is None or rerun.name != channel:
                    msg = ('PogChamp You\'ve been voted into the #%s '
                           'Community Spotlight. If you don\'t want the honor '
                           'drop into chat on the directory page '
                           'and say !removeme. I\'ll pick someone else!'
                           % community.name)
                    self.proxy(conn, '#%s' % channel, msg, community)
            else:
                self.start_vote(conn, community)
        except TwitchApiError:
            self.say(conn, community.channel,
                     'I\'m having some issues, can you try again?')

    def remove_broadcaster(self, conn, community, user, validate_user=True):
        '''
        This command will remove the current promoted channel from the
        community, but only if the broadcaster calls it
        '''
        if not validate_user or community.promoted_channel == user:
            self.set_broadcaster(conn, community, user, None)
            self.proxy(conn, '#%s' % user, 'You\'ve been removed from the %s '
                       'Community Spotlight.' % community.name, community)

    @command_throttle(seconds=10)
    def set_topic(self, conn, community, user, args):
        '''
        Set the About Us title for this community
        '''
        title = ' '.join(args)
        try:
            TwitchApi.set_title(community.channel[1:], title)
            self.say(conn, community.channel, 'I set the title to %s' % title)
            self.send_pubsub(topic='channel.%s' % community.channel[1:],
                             message={'type': 'status', 'status': title})
        except TwitchApiError:
            self.say(conn, community.channel, 'I tried to set the title but '
                     'then decided I had better things to do')

    def request_time(self, conn, community, user):
        '''
        Activates request_time for a given channel
        '''
        community.is_request_time = True
        self.say(conn, community.channel, 'Time to take some requests')

    def request_time_over(self, conn, community, user):
        '''
        Deactivates request_time for a given channel
        '''
        community.is_request_time = False
        self.say(conn, community.channel, 'Alright no more requests for a bit')

    @command_throttle(seconds=600)
    def request(self, conn, community, user, repeat_every=30,
                end_after=DEFAULT_VOTE_DURATION.seconds):
        '''
        Start vote immeadietly
        '''
        try:
            repeat_every = datetime.timedelta(seconds=int(repeat_every))
            end_after = datetime.timedelta(seconds=int(end_after))
            self.start_vote(conn, community, repeat_every, end_after)
        except ValueError:
            self.say(conn, community.channel,
                     'I don\'t understand what you want. Maybe try !help')

    def request_over(self, conn, community, user, ignore_results=False):
        '''
        End the current vote and set the winner as the promoted channel
        '''
        if ignore_results:
            # end the vote, but do not finalize and set winner
            community.end_vote()
            return
        self.end_vote(conn, community)

    @command_throttle(seconds=300)
    def no_channels_live(self, conn, community, user):
        self.say(conn, community.channel, 'No channels live! Maybe later?')
        log = '[%s] No live channels, cannot start vote' % community.name
        logger.rollbar(log, logger.WARNING)

    def start_vote(self, conn, community, repeat_every=None, end_after=None):
        try:
            vote = community.start_vote(repeat_every, end_after)
            self.say(conn, community.channel, 'Emote your vote! Choose the '
                     'broadcaster you want to see by chatting their emote or '
                     'channel name!')
            pubsub_message = {
                'type': 'request-start',
                'channels': vote.channels
            }
            self.send_community_pubsub(community, pubsub_message)
        except NoLiveChannels:
            self.no_channels_live(conn, community, None)
        except VoteAlreadyInProgressError:
            self.say(conn, community.channel, 'Naw, we\'re already voting')

    def end_vote(self, conn, community):
        '''
        Finalizes a vote. This will send the results as well as
        set the winner as the promoted channel for the community.
        '''
        # display results to chat
        human_winner = community.vote.human_friendly_winner
        message = 'The vote is finished! Winner is %s' % human_winner
        self.say(conn, community.channel, message)
        # update systems with new promoted channel
        winner = community.vote.winner
        self.set_broadcaster(conn, community, None, winner)
        # notify consumers that vote is over
        pubsub_message = {'type': 'request-over', 'winner': winner}
        self.send_community_pubsub(community, pubsub_message)
        community.end_vote()

    def next_vote(self, community):
        '''
        Return the time of the next planned vote or None if there is not
        one planned
        '''
        if not community.last_vote_at:
            return None
        between = datetime.timedelta(seconds=SETTINGS['time_between_votes'])
        next_vote_time = community.last_vote_at + between
        return next_vote_time

    def is_time_for_vote(self, community):
        if community.is_request_time is False:
            return False
        next_vote = self.next_vote(community)
        if next_vote is None:
            return True
        return next_vote < datetime.datetime.now()

    def handle_request_times(self, conn, community):
        '''
        Run loop for requests. Handles request_times and vote announcements
        '''
        if self.is_time_for_vote(community):
            self.start_vote(conn, community)
        if community.vote is not None:
            if community.is_vote_completed:
                self.end_vote(conn, community)
            elif community.vote.is_time_to_announce:
                for message in community.vote.human_friendly_options:
                    self.say(conn, community.channel, message)
                community.vote.update_last_vote_announcement()

    def update_promoted_channel(self, conn, community):
        try:
            api_community = TwitchApi.get_community(community.name)
            community.promoted_channel = api_community.promoted_channel
            if community.promoted_channel is None and community.vote is None:
                self.start_vote(conn, community)
        except TwitchApiError:
            log = '[%s] could not update promoted channel' % community.name
            logger.rollbar(log, logger.WARNING)
        except NoLiveChannels:
            log = '[%s] has no live channels, cannot vote' % community.name
            logger.rollbar(log, logger.WARNING)

    def tell_chat_whats_next(self, conn, community):
        if community.is_request_time and community.vote is None:
            self.whats_next(conn, community, None)

    def update_dashboard(self, conn, community):
        '''
        Updates grafana dashboard on
        https://grafana.prod.us-west2.justin.tv/dashboard/db/communities-dashboard
        '''
        self.update_chatters(conn, community)
        self.update_stream_count(conn, community)

    def update_stream_count(self, conn, community):
        '''
        Updates Stream count in grafana for community
        '''
        stat = ".".join(["stream_count", community.name])
        try:
            _, count = TwitchApi.get_streams_by_community(community.name)
            gauge(stat, count)
        except TwitchApiError:
            pass

    def update_chatters(self, conn, community):
        '''
        Updates Chatters count in grafana for community
        '''
        stat = ".".join(["chatters", community.name])
        try:
            _, count = TwitchApi.get_chatters(community.channel[1:])
            gauge(stat, count)
        except TwitchApiError:
            pass
