import datetime
import sys
import communitybot.logger as logger
from communitybot.bots.twitch import TwitchBot
from communitybot.clients.mission_control import (
    MissionControlClient,
    MissionControlClientError
)
from communitybot.clients.twitch import TwitchApi, TwitchApiError
from communitybot.settings import SETTINGS
from communitybot.util import throttle
from communitybot.vote import StreamVote


def get_mission_control_client(channel):
    '''
    Return mission control client for the provided channel
    '''
    host = SETTINGS['mission_control_api']
    if channel.startswith('#'):
        channel = channel[1:]
    return MissionControlClient(host=host, channel=channel)


class Command(object):

    def __init__(self, cmd, method, usage=None):
        self.cmd = cmd
        self.method = method
        self.usage = usage


class VoteAlreadyInProgressError(Exception):
    def __str__(self):
        return repr('Vote is already in progress')


class NoVoteInProgressError(Exception):
    def __str__(self):
        return repr('No Options Available')


class VoteStore:

    VOTES = {}
    LOGS = {}

    @staticmethod
    def get_vote(channel):
        return VoteStore.VOTES.get(channel, None)

    @staticmethod
    def add_vote(channel):
        if VoteStore.get_vote(channel) is not None:
            raise VoteAlreadyInProgressError()
        emotes = SETTINGS['emotes']
        VoteStore.VOTES[channel] = StreamVote(channel[1:], emotes)
        VoteStore.LOGS[channel] = datetime.datetime.now()
        return VoteStore.VOTES[channel]

    @staticmethod
    def end_vote(channel):
        if VoteStore.get_vote(channel) is None:
            raise NoVoteInProgressError()
        del VoteStore.VOTES[channel]


class StreamBot(TwitchBot):
    '''
    Stream Bot supports managing a stream channel. It allows users to vote
    on the next show they want and give information about each show.
    '''

    COMMANDS = [
        Command('!help', 'help', '!help will give this messsage. Do !help '
                '!command to see help for specific commands'),
        Command('!whatsplaying', 'whatsplaying', '!whatsplaying will tell you '
                'about the currently playing episode'),
        Command('!whatsnext', 'whatsnext', '!whatsnext will tell you the next '
                'show in the queue'),
    ]
    MOD_COMMANDS = []
    ADMIN_COMMANDS = [
        Command('!startstream', 'startstream', '!startstream will start an '
                'indefinite stream on the current channel'),
        Command('!stopstream', 'stopstream', '!startstream will cancel '
                'the current stream'),
        Command('!requesttime', 'requesttime', '!requesttime will turn on '
                'user voting during every episode to select the next'),
        Command('!requesttimeover', 'requesttimeover', '!requesttimeover will '
                'end the current request time'),
        Command('!request', 'startrequest', '!request will start a new vote '
                'from random content to add to the queue'),
        Command('!requestover', 'endrequest', '!requestover will end the '
                'current vote and add the winner to play next'),
    ]

    START_BUFFER = datetime.timedelta(seconds=60)
    END_BUFFER = datetime.timedelta(minutes=2)
    MAX_VOTE_DURATION = datetime.timedelta(minutes=5)

    has_scheduled_request = False
    is_request_time = False

    def parse_command(self, msg, commands):
        '''
        Parse a message searching for a known (and supported) command
        '''
        if msg.startswith('!'):
            parts = msg.split()
            command, arguments = parts[0].lower(), parts[1:]
            handler = next((c for c in commands if c.cmd == command), None)
            if handler is not None:
                return getattr(self, handler.method), arguments
        return None, None

    def get_available_commands(self, user, channel):
        '''
        Return a list of available commands for the provided user
        '''
        commands = self.COMMANDS[:]
        if self.is_user_moderator(user, channel):
            commands += self.MOD_COMMANDS[:]
        if self.is_user_admin(user):
            commands += self.ADMIN_COMMANDS[:]
        return commands

    def on_pubmsg(self, conn, event):
        '''
        Track each message and parse for commands
        '''
        try:
            user, channel, msg = self.get_user_msg_from_irc(event)
            commands = self.get_available_commands(user, channel)
            handler, args = self.parse_command(msg, commands)
            if handler:
                return handler(conn, user, channel, *args)
            vote = VoteStore.get_vote(channel)
            if vote is not None and vote.is_valid_vote(msg):
                return vote.vote(user, msg)
        except Exception as e:
            logger.exception(sys.exc_info())

    # BOT COMMANDS

    @throttle(seconds=10)
    def help(self, conn, user, channel, command=None, *args):
        commands = self.get_available_commands(user, channel)
        if command:
            if command.startswith('!') is None:
                command = '!%s' % command
            help_command, _ = self.parse_command(command, commands)
            if help_command:
                self.say(conn, channel, help_command.usage)
            else:
                self.say(conn, channel, 'I\'m afraid I can\'t do that')
            return
        command_list = ', '.join([c.cmd for c in commands])
        self.say(conn, channel, 'Yo %s you can do %s' % (user, command_list))
        self.say(conn, channel, 'If you need help with something specific you '
                 'can do `!help <command>`')

    @throttle(seconds=10)
    def whatsplaying(self, conn, user, channel, *args):
        '''
        Send the title of the current playing video
        '''
        client = get_mission_control_client(channel)
        stream = client.get_current_stream()
        if stream and stream.current_video:
            title = (stream.current_video.video.season + ' ' +
                     stream.current_video.video.title)
            title = title.replace('_', ' ')
            title = title.replace('.mp4', '')
            self.say(conn, channel, 'Now playing "%s"' % title)
        else:
            self.say(conn, channel, 'Nothing, right now')

    @throttle(seconds=10)
    def whatsnext(self, conn, user, channel, *args):
        '''
        Send the title of the next video in the playlist
        '''
        client = get_mission_control_client(channel)
        stream = client.get_current_stream()
        if stream and stream.next_video:
            title = (stream.next_video.video.season + ' ' +
                     stream.next_video.video.title)
            title = title.replace('_', ' ')
            title = title.replace('.mp4', '')
            self.say(conn, channel, 'Will play "%s" next' % title)
        elif self.is_request_time:
            if self.next_request_at:
                time_to_vote = self.next_request_at - datetime.datetime.now()
                msg = 'The people will decide in about %d minutes!' % (
                    int(time_to_vote.seconds / 60))
                self.say(conn, channel, msg)
                return
            self.say(conn, channel, 'I will let the people decide!')
        else:
            self.say(conn, channel, 'Still thinking about it')

    def startstream(self, conn, user, channel, *args):
        '''
        Start an indefinite stream on the provided channel
        '''
        client = get_mission_control_client(channel)
        stream = client.get_current_stream()
        if stream:
            self.say(conn, channel, 'We\'re already on the road - buckle up!')
            return
        stream = client.create_stream(start_at=datetime.datetime.now(),
                                      indefinite=True)
        self.say(conn, channel, 'Buckle up, we\'re getting started!')

    def stopstream(self, conn, user, channel, *args):
        '''
        Stop the stream on provided channel
        '''
        client = get_mission_control_client(channel)
        stream = client.get_current_stream()
        if stream:
            stream = client.stop_stream(stream)
            self.say(conn, channel, 'Thanks for watching, see you soon!')
        else:
            self.say(conn, channel, 'Wut? We\'re not streaming.')

    def startrequest(self, conn, user, channel, repeat_every=30,
                     end_after=MAX_VOTE_DURATION.seconds, *args):
        '''
        Start a vote that will allow users to vote for the next
        resource to play
        '''
        try:
            repeat_every = int(repeat_every)
            if end_after is not None:
                end_after = int(end_after)
        except ValueError:
            self.say(conn, channel, 'Sorry? Don\'t think so!')
        client = get_mission_control_client(channel)
        stream = client.get_current_stream()
        vote = VoteStore.get_vote(channel)
        if stream is None:
            self.say(conn, channel, 'Naw, no stream at the moment.')
            return
        if vote:
            self.say(conn, channel, 'Already in a vote hey.')
            return
        vote = VoteStore.add_vote(channel)

        def say_vote():
            '''
            Tell chat the current vote standings
            '''
            if VoteStore.get_vote(channel) == vote:
                self.say(conn, channel, 'TIME TO VOTE!')
                for o in vote.options:
                    self.say(conn, channel, '%s %s' % (o.emote, str(o)))
                self.reactor.execute_delayed(repeat_every, say_vote)

        say_vote()
        if end_after is not None:
            args = (conn, user, channel)
            self.reactor.execute_delayed(end_after, self.endrequest, args)

    def endrequest(self, conn, user, channel, *args):
        '''
        End the current vote and enqueue the winning resource
        '''
        client = get_mission_control_client(channel)
        stream = client.get_current_stream()
        vote = VoteStore.get_vote(channel)
        if vote is None:
            self.say(conn, channel, 'Wait, what vote? Ain\'t no vote here')
        VoteStore.end_vote(channel)
        winner = vote.winner
        client.add_video_to_stream(stream, winner.resource)
        msg = 'THE VOTE IS DONE! The winner is %s' % winner
        self.say(conn, channel, msg)
        self.has_scheduled_request = False

    def requesttime(self, conn, user, channel, *args):
        '''
        Will start running requests during each episode to select the next
        resource to watch.
        '''
        if self.is_request_time:
            self.say(conn, channel, 'Already on it boss')
            return
        self.is_request_time = True
        self.say(conn, channel, 'Time to take some requests')
        self.reactor.execute_delayed(1, self._handle_requesttime, (channel,))

    def requesttimeover(self, conn, user, channel, *args):
        '''
        Deactivates the current reuqest time polling
        '''
        if self.is_request_time is False:
            self.say(conn, channel, 'Cool... we\'re not doing requests anyway')
            return
        self.is_request_time = False
        self.say(conn, channel, 'Alright no more requests for a bit')

    def _handle_requesttime(self, channel):
        '''
        Run loop that recurses unless request time is over
        '''
        if self.is_request_time is False:
            return
        client = get_mission_control_client(channel)
        stream = client.get_current_stream()
        if self.has_scheduled_request is False and stream.next_video is None:
            self.schedule_request(channel, stream.current_video)
        self.reactor.execute_delayed(1, self._handle_requesttime, (channel,))

    def schedule_request(self, channel, current_video):
        '''
        Schedule a request during the current video for the next video if
        possible
        '''
        if current_video.time_left < datetime.timedelta(minutes=3):
            return
        length = current_video.time_left - self.END_BUFFER - self.START_BUFFER
        duration = min(self.MAX_VOTE_DURATION, length)
        end_at = current_video.time_left - self.END_BUFFER
        start_at = end_at - duration
        args = (self.connection, None, channel, 30, duration.seconds)
        self.has_scheduled_request = True
        self.next_request_at = datetime.datetime.now() + start_at
        self.say(self.connection, channel,
                 'Next vote in about %d minutes' % int(start_at.seconds / 60))
        self.reactor.execute_delayed(start_at.seconds, self.startrequest, args)
