from __future__ import print_function
from irc.client import ServerNotConnectedError
from irc.bot import SingleServerIRCBot
from communitybot.clients.pubsub import PubsubClient, FailedRequest
from communitybot.settings import SETTINGS
import communitybot.logger as logger

SERVER = 'irc.twitch.tv'
PORT = 6667


class LoggingBot(SingleServerIRCBot):
    '''
    IRC Bot that logs it's health
    '''

    def _on_disconnect(self, c, e):
        super(LoggingBot, self)._on_disconnect(c, e)
        logger.rollbar('Connection lost', logger.ERROR)


class ConnectionService:
    '''
    Makes a number of IRC connections and allows messages to be sent in a
    round-robin fashion to each one.
    '''

    def __init__(self, nickname=None, servers=[], count=1):
        self.nickname = nickname
        self.servers = servers
        self.connections = [self.new_connection() for _ in range(count)]

    def new_connection(self):
        '''
        Make a new connection to IRC
        '''
        conn = LoggingBot(self.servers, self.nickname, self.nickname)
        conn._connect()
        return conn

    def recycle_connection(self, conn):
        '''
        Replace a connection from our pool with a new one
        '''
        try:
            index = next(i for i, c in enumerate(self.connections)
                         if conn == c.connection)
            self.connections.pop(index)
            self.connections.append(self.new_connection())
        except StopIteration:
            pass

    def next_connection(self):
        '''
        Return the next connection in round-robin fashion
        '''
        if len(self.connections) < 1:
            logger.rollbar('No connections available', logger.ERROR)
            self.connections.append(self.new_connection())
            return conn
        self.rotate_connections()
        conn = self.connections[0].connection
        if conn.is_connected() is False:
            logger.rollbar('Connection is not connected', logger.ERROR)
            return self.next_connection()
        return conn

    def rotate_connections(self):
        '''
        Rotate the connections to put the last used connection
        at the end of the list
        '''
        self.connections = self.connections[1:] + self.connections[:1]

    def say(self, channel, msg, conn=None):
        '''
        Sends message to channel from the next IRC connection
        '''
        if conn is None:
            conn = self.next_connection()
            try:
                conn.privmsg(channel, '/me %s' % msg)
            except ServerNotConnectedError:
                logger.rollbar('Connection not connected', logger.WARNING)
                self.recycle_connection(conn)
                return self.say(channel, msg)
            return
        try:
            conn.privmsg(channel, '/me %s' % msg)
        except ServerNotConnectedError:
            logger.rollbar('Connection not connected on provided connection',
                           logger.WARNING)
            return self.say(channel, msg)


class TwitchBot(SingleServerIRCBot):
    '''
    Generalized Twitch bot for interacting with a channel via
    TMI/IRC Twitch Chat
    '''

    def __init__(self, nickname=None, password=None, server=SERVER,
                 port=PORT, channels=[]):
        servers = [(server, port, password)]
        super(TwitchBot, self).__init__(servers, nickname, nickname)
        self.nickname = nickname
        self.connection_service = ConnectionService(nickname, servers, 10)
        self._channels = channels
        self.mods = dict([(c, []) for c in self._channels])

    def on_welcome(self, conn, event):
        '''
        On server connection, join requested channels
        '''
        conn.send_raw('CAP REQ :twitch.tv/membership')
        conn.send_raw('CAP REQ :twitch.tv/commands')
        for channel in self._channels:
            conn.join(channel)

    def on_all_raw_messages(self, conn, event):
        '''
        When in debug mode print all raw IRC messages
        '''
        if SETTINGS['debug']:
            print(event)

    def on_pubnotice(self, conn, event):
        '''
        Watch for MOD status on pubnotice
        '''
        channel = event.target
        msg = event.arguments[0]
        mods = []
        if msg.startswith('The moderators of this room are: '):
            mods = [m.strip() for m in msg[33:].split(',')]
        self.set_mods(channel, mods)

    def on_join(self, conn, event):
        '''
        Track status of each joined channel
        '''
        self.request_mods(conn, event.target)
        self.reactor.execute_every(30, self.request_mods, (conn, event.target))

    def parse_command(self, msg, commands):
        '''
        Parse a message searching for a known (and supported) command
        '''
        if msg.startswith('!'):
            parts = msg.split()
            command, arguments = parts[0].lower(), parts[1:]
            if command in commands:
                return command, arguments
        return None, None

    def is_user_admin(self, user):
        '''
        Return True if the user is an admin for this bot
        '''
        return user in SETTINGS['admins']

    def is_user_moderator(self, user, channel):
        '''
        Return True if user is a moderator of the current channel
        '''
        return user in self.mods.get(channel, [])

    def set_mods(self, channel, mods=[]):
        '''
        Update the moderators for provided channel
        '''
        self.mods[channel] = mods

    def is_moderator(self, channel):
        return self.nickname in self.mods.get(channel, [])

    def say(self, conn, channel, msg):
        '''
        Sends message from bot to channel
        '''
        if channel.startswith('#') is False:
            print('why not start with #')
            channel = '#%s' % channel
        if SETTINGS['debug']:
            print('[%s]: %s' % (channel, msg.encode('ascii', 'ignore')))
        # when you are moderator, use the provided connection for better
        # service, otherwise use the connection_service provider
        if self.is_moderator(channel):
            self.connection_service.say(channel, msg, conn)
        else:
            self.connection_service.say(channel, msg)

    def timeout(self, conn, user, channel, length):
        '''
        Timeout a user from the channel for provided length of time
        '''
        conn.privmsg(channel, '/timeout %s %d' % (user, length.seconds))

    def ban(self, conn, user, channel):
        '''
        Ban provided user from the channel
        '''
        conn.privmsg(channel, '/ban %s' % user)

    def request_mods(self, conn, channel):
        conn.privmsg(channel, '/mods')

    def get_user_msg_from_irc(self, event):
        '''
        Return a tuple of (username, message) from an IRC event
        '''
        return event.source.split('!')[0], event.target, event.arguments[0]

    def send_pubsub(self, topic, message):
        '''
        Send pubsub notification for some event
        '''
        client = PubsubClient(host=SETTINGS['pubsub_host'])
        try:
            client.publish(topics=[topic], message=message)
        except FailedRequest:
            log = 'Pubsub failed to send [%s: %s]' % (topic, message)
            logger.rollbar(log, logger.ERROR)
