import json
import requests
from datetime import datetime, timedelta


class MissionControlClientError(IOError):
    '''Generalized mission control request failure'''

    def __init__(self, body):
        self.body = body

    def __str__(self):
        return repr(self.body)


class ResourceNotFoundError(MissionControlClientError):
    '''Requested resource does not exist - 404'''


class Video(object):
    '''
    Representation of a video resource from Mission Control. Each channel
    has a number of video resources that can be enqueued and streamead to that
    channel during a stream. This is the video resource, unrelated to a stream.
    '''

    def __init__(self, id, title, show, season, length, deleted_at=None,
                 **kwargs):
        self.id = id
        self.title = title
        self.show = show
        self.season = season
        self.length = length
        self.deleted_at = deleted_at

    def __eq__(self, b):
        return self.id == b.id

    def __str__(self):
        return '%s / %s / %s' % (self.show, self.season, self.title)

    @property
    def name(self):
        return self.id

    @property
    def display_name(self):
        return '%s - %s' % (self.show, self.title)

    @property
    def display_name(self):
        return '%s - %s' % (self.show, self.title)

    @staticmethod
    def __api__(json):
        '''
        Deserialize API response JSON to object
        '''
        return Video(**json)

    @property
    def is_deleted(self):
        '''
        Return True when this resource has been deleted and cannot be played
        '''
        return self.deleted_at is not None


class StreamVideo(object):
    '''
    Representation of a stream playable item from Mission Control. A stream
    has a list of playable StreamVideos which contain a reference to the
    video resource as well as related positional and temporal meta data.
    '''

    FINISHED = 'finished'
    STREAMING = 'streaming'
    UNSTREAMED = 'unstreamed'

    def __init__(self, id, state, video, seconds_streamed, time_left,
                 **kwargs):
        self.id = id
        self.state = state
        self.video = video
        self.seconds_streamed = timedelta(seconds=seconds_streamed)
        self.time_left = timedelta(seconds=time_left)

    def __eq__(self, b):
        return self.id == b.id

    def __str__(self):
        return '[%s] %s' % (self.state, self.video)

    @staticmethod
    def __api__(json):
        '''
        Deserialize API response JSON to object
        '''
        video = Video.__api__(json.pop('video'))
        return StreamVideo(video=video, **json)


class Stream(object):
    '''
    Representation of a stream from Mission Control. A stream is a playlist
    of resource videos as well as an assigned channel and start time.
    '''

    CANCELED = 'canceled'
    FINISHED = 'finished'
    SCHEDULED = 'scheduled'
    STARTED = 'started'

    def __init__(self, id, channel, state, videos, start_time,
                 current_video, next_video, **kwargs):
        self.id = id
        self.channel = channel
        self.state = state
        self.videos = videos
        self.start_time = start_time
        self.current_video = current_video
        self.next_video = next_video

    def __eq__(self, b):
        return self.id == b.id

    def __str__(self):
        return '[%s] %s - %s' % (self.channel, self.state, self.start_time)

    @staticmethod
    def __api__(json):
        '''
        Deserialize API response JSON to object
        '''
        channel = json['channel']['name']
        stream_videos = json['stream_session_videos']
        videos = [StreamVideo.__api__(v) for v in stream_videos]
        current_video = None
        next_video = None
        try:
            current_id = json['current_stream_session_video']['id']
            current_video = next(v for v in videos if v.id == current_id)
        except KeyError, StopIteration:
            pass
        try:
            next_id = json['next_stream_session_video']['id']
            next_video = next(v for v in videos if v.id == next_id)
        except KeyError, StopIteration:
            pass
        return Stream(id=json['id'], channel=channel, state=json['state'],
                      videos=videos, start_time=json['start_time'],
                      current_video=current_video, next_video=next_video)


class MissionControlClient(object):
    '''
    The Client class allows communication with Mission Control to work out
    the current state of a channel stream.

    Usage:

        client = MissionControlClient(host='http://url.to.api', channel='xxx')
        stream = client.get_current_stream()
        video = client.get_videos(limit=1, random=True)[0]
        stream = client.add_video_to_stream(stream, video)
    '''

    def __init__(self, host, channel):
        self.host = host
        self.channel = channel

    @staticmethod
    def get(url, params={}):
        '''
        Make GET HTTP request and parse response with validation and
        consistnt error handling.
        '''
        resp = requests.get(url, params=params)
        return MissionControlClient.parse_body(resp)

    @staticmethod
    def put(url, data={}):
        '''
        Make PUT HTTP request and parse response with validation and
        consistnt error handling.
        '''
        resp = requests.put(url, data=data)
        return MissionControlClient.parse_body(resp)

    @staticmethod
    def post(url, data={}):
        '''
        Make POST HTTP request and parse response with validation and
        consistnt error handling.
        '''
        resp = requests.post(url, data=data)
        return MissionControlClient.parse_body(resp)

    @staticmethod
    def delete(url, params={}):
        '''
        Make DELETE HTTP request and parse response with validation and
        consistnt error handling.
        '''
        resp = requests.delete(url, params=params)
        return MissionControlClient.parse_body(resp)

    @staticmethod
    def parse_body(resp):
        '''
        Raises if the status code is bad, otherwise returns the JSON body
        as a dictionary.
        '''
        if resp.status_code == 404:
            raise ResourceNotFoundError(resp)
        if resp.status_code != 200:
            raise MissionControlClientError(resp)
        try:
            body = resp.json()
        except ValueError:
            raise MissionControlClientError(resp)
        return body

    def create_stream(self, start_at=None, videos=[], indefinite=False):
        '''
        Create and possibly start a new stream to the current channel.

        POST | PUT /channels/:channel/stream_sessions
        - start_time: isoformat() when to start stream
        - indefinite: continue to stream 24/7 randomly if nothing is provided
        '''
        url = '%s/channels/%s/stream_sessions' % (self.host, self.channel)
        start_time = start_at and start_at.isoformat() or datetime.now()
        data = {
            'start_time': start_time,
            'videos': [v.id for v in videos],
            'indefinite': str(indefinite).lower()
        }
        body = MissionControlClient.post(url, data)
        return Stream.__api__(body['stream_session'])

    def get_current_stream(self):
        '''
        Return the current stream (or None) for provided channel

        GET /channels/:channel/stream_sessions/current
        '''
        url = '%s/channels/%s/stream_sessions/current' % (
            self.host, self.channel)
        try:
            body = MissionControlClient.get(url)
            return Stream.__api__(body['stream_session'])
        except ResourceNotFoundError:
            return None

    def get_stream(self, id):
        '''
        Return a stream by ID. Will raise ResourceNotFoundError if a
        resource with matching ID is not found.

        GET /channels/:channel/stream_sessions/:stream
        '''
        url = '%s/channels/%s/stream_sessions/%d' % (
            self.host, self.channel, id)
        body = MissionControlClient.get(url)
        return Stream.__api__(body['stream_session'])

    def start_stream(self, stream, start_at, indefinite=False):
        '''
        Create and possibly start a new stream to the current channel.
        Will raise ResourceNotFoundError if a resource with matching ID
        is not found.

        POST | PUT /channels/:channel/stream_sessions
        - start_time: isoformat() when to start stream
        - indefinite: continue to stream 24/7 randomly if nothing is provided
        '''
        url = '%s/channels/%s/stream_sessions/%d' % (
            self.host, self.channel, stream.id)
        start_time = start_at and start_at.isoformat() or datetime.now()
        data = {
            'start_time': start_time,
            'indefinite': str(indefinite).lower()
        }
        body = MissionControlClient.put(url, data)
        return Stream.__api__(body['stream_session'])

    def stop_stream(self, stream):
        '''
        Will stop the provided stream, if currently being consumed.

        DEL /channels/:channel/stream_sessions/:stream
        '''
        url = '%s/channels/%s/stream_sessions/%d' % (
            self.host, self.channel, stream.id)
        body = MissionControlClient.delete(url)
        return Stream.__api__(body['stream_session'])

    def add_video_to_stream(self, stream, video):
        '''
        Add a new video to the provided stream. Can prepend or append.

        POST '/channels/:channel/stream_sessions/:stream_id/videos
        - id: video id
        '''
        url = '%s/channels/%s/stream_sessions/%d/videos' % (
            self.host, self.channel, stream.id)
        params = {'id': video.id}
        body = MissionControlClient.post(url, params)
        return Stream.__api__(body['stream_session'])

    def get_video(self, id):
        '''
        Return a video by ID. Will raise ResourceNotFoundError if a
        resource with matching ID is not found.

        GET /channels/:channel/videos/:video
        '''
        url = '%s/channels/%s/videos/%d' % (self.host, self.channel, id)
        body = MissionControlClient.get(url)
        return Video.__api__(body['video'])

    def get_videos(self, limit=1, random=True, exclude_videos=[]):
        '''
        Return list of video resources available within provided channel.
        List can be randomized by mission control to provide a sane set
        or results based on existing plays.

        GET '/channels/:channel/videos
        - limit: number of resources to return
        - random: true if randomized
        - exclude_videos: a list of videos to exclude from returned list
        '''
        url = '%s/channels/%s/videos' % (self.host, self.channel)
        params = {
            'limit': limit,
            'random': str(random).lower(),
            'exclude_videos': [v.id for v in exclude_videos]
        }
        body = MissionControlClient.get(url, params)
        return [Video.__api__(v) for v in body['videos']], body['_total']
