import json
import requests


class FailedRequest(IOError):
    '''Generalized pubsub request failure'''


class InvalidRequest(FailedRequest):
    '''Invalid pubsub request'''


class InvalidHost(FailedRequest):
    '''Invalid pubsub host provided'''


class PubsubClient:
    '''
    The Client class allows messages to be sent to a specific
    pubsub client for notifying consumer clients.

    Usage:

        client = Client(host="http://pubsub-broker.internal.twitch.tv")
        topic = "creative-community.painting"
        message = {"type": "vote-over", "winner": "food"}
        client.publish(topics=[topic], message=message)
    '''

    def __init__(self, host):
        self.host = host

    def publish(self, topics, message):
        '''
        Send message to pubsub broker in correct format
        '''
        headers = {"Content-Type": "json"}
        data = {
            "topics": topics,
            "message": json.dumps(message)
        }
        try:
            r = requests.post(self.host + "/v1/message", data=json.dumps(data),
                              headers=headers)
            if r.status_code == 400:
                raise InvalidRequest()
        except (requests.exceptions.URLRequired,
                requests.exceptions.MissingSchema):
            raise InvalidHost()
