import requests
import json
import sys
from communitybot.settings import SETTINGS
import communitybot.logger as logger
from requests_toolbelt import MultipartEncoder

API_HOST = SETTINGS["twitch_api"]
TMI_HOST = SETTINGS["tmi_api"]
PROBLEMATIC_STATUS_CODES = [401, 403, 404]


class ApiChannel:

    def __init__(self, name, display_name, status, mature, url, **kwargs):
        self.name = name
        self.display_name = display_name
        self.status = status
        self.mature = mature
        self.url = url

    def __str__(self):
        return self.name


class ApiStream:

    def __init__(self, channel, game, viewers, created_at, **kwargs):
        self.channel = channel
        self.game = game
        self.viewers = viewers
        self.created_at = created_at

    def __str__(self):
        return '%s playing %s with %d viewers' % (
            self.channel.name, self.game, self.viewers)


class ApiCommunity:

    def __init__(self, name, game, promoted_channel=None, **kwargs):
        self.name = name
        self.game = game
        self.promoted_channel = promoted_channel

    def __str__(self):
        return self.name


class TwitchApiError(Exception):
    def __init__(self, body):
        self.body = body

    def __str__(self):
        return repr(self.body)


class TwitchApi:

    @staticmethod
    def default_headers(headers={}):
        default = {
            'Authorization': 'OAuth %s' % SETTINGS['oauth_token'],
            'Accept': 'application/vnd.twitchtv.v2+json',
            "Content-Type": 'application/json'
        }
        default.update(headers)
        return default

    @staticmethod
    def get(*args, **kwargs):
        '''
        Proxy GET request and return valid body
        '''
        resp = requests.get(*args, **kwargs)
        TwitchApi.log_response(resp)
        return TwitchApi.parse_body(resp)

    @staticmethod
    def put(*args, **kwargs):
        '''
        Proxy PUT request and return valid body
        '''
        resp = requests.put(*args, **kwargs)
        TwitchApi.log_response(resp)
        return TwitchApi.parse_body(resp)

    @staticmethod
    def set_title(channel, status):
        '''
        Sets the title for a given channel
        '''
        data = json.dumps({"channel": {"status": status}})
        url = "%s/channels/%s" % (API_HOST, channel)
        headers = TwitchApi.default_headers()
        resp = TwitchApi.put(url, headers=headers, data=data)
        return ApiChannel(**resp)

    @staticmethod
    def get_streams_by_community(community):
        '''
        Returns a list of streams on a given community
        '''
        url = "%s/streams/communities/%s?game=Creative&limit=100" % (
            API_HOST, community)
        resp = TwitchApi.get(url, headers=TwitchApi.default_headers())
        streams = []
        try:
            for s in resp['streams']:
                _channel = s.pop('channel')
                channel = ApiChannel(**_channel)
                streams.append(ApiStream(channel=channel, **s))
        except KeyError:
            logger.log("Malformed response from streams API: %s" % resp,
                       logger.ERROR)
            raise TwitchApiError(resp)
        return streams, resp['_total']

    @staticmethod
    def get_community(community):
        '''
        Returns details about a community
        '''
        url = "%s/games/Creative/communities/%s" % (API_HOST, community)
        resp = TwitchApi.get(url, headers=TwitchApi.default_headers())
        return ApiCommunity(**resp['community'])

    @staticmethod
    def set_promoted_channel(community, promoted_channel):
        '''
        Sets a community's promoted_channel
        '''
        payload = MultipartEncoder(dict(promoted_channel=promoted_channel))
        url = "%s/games/Creative/communities/%s" % (API_HOST, community)
        custom_headers = {'Content-Type': payload.content_type}
        headers = TwitchApi.default_headers(custom_headers)
        data = payload.to_string()
        resp = TwitchApi.put(url, data=data, headers=headers)
        return ApiCommunity(**resp)

    @staticmethod
    def get_chatters(channel):
        '''
        Gets a channels's chatters
        '''
        url = "%s/group/user/%s/chatters" % (TMI_HOST, channel)
        resp = TwitchApi.get(url)
        return resp['chatters'], resp['chatter_count']

    @staticmethod
    def log_response(resp):
        '''
        Logs a response if it has a problematic status code
        '''
        if resp.status_code in PROBLEMATIC_STATUS_CODES:
            logger.log("%s failed with %d" % (resp.url, resp.status_code),
                       logger.ERROR)

    @staticmethod
    def parse_body(resp):
        '''
        Raises if the status code is bad, otherwise returns the body
        '''
        try:
            body = resp.json()
        except ValueError:
            raise TwitchApiError(resp)
        if resp.status_code != 200:
            raise TwitchApiError(resp)
        return body
