from __future__ import print_function
from communitybot.settings import SETTINGS
from communitybot.vote import CommunityVote, NoOptionsAvailable
from communitybot.tracking import track
import datetime


class CommunityException(Exception):
    pass


class NoLiveChannels(CommunityException):
    def __str__(self):
        return repr('No live channels for this community')


class VoteAlreadyInProgressError(CommunityException):
    def __str__(self):
        return repr('Vote is already in Progress')


class NoVoteInProgress(CommunityException):
    def __str__(self):
        return repr('No Vote in Progress')


class Community:

    def __init__(self, name, channel, emotes, mods=[], promoted_channel=None):
        self.name = name
        self.channel = channel
        self.emotes = emotes
        self.mods = mods
        self.promoted_channel = promoted_channel

        self.is_tracking = False
        self.vote = None
        self.last_vote_at = None
        self.is_request_time = False

    def __eq__(self, other):
        return self.name == other.name

    def is_admin(self, user):
        '''
        Return True when if the user is an admin of the channel
        '''
        return self.channel[1:] == user or user in SETTINGS['admins']

    def is_mod(self, user):
        '''
        Return True if the user is a mod of the channel
        '''
        return user in self.mods or user in SETTINGS['admins']

    @property
    def bot_is_moderator(self):
        '''
        Return True if the bot is a moderator
        '''
        return self.is_mod(SETTINGS['nickname'])

    def start_vote(self, repeat_every=None, end_after=None):
        '''
        Starts a vote
        '''
        if self.vote:
            raise VoteAlreadyInProgressError()
        try:
            self.vote = CommunityVote(self.name, self.emotes,
                                      repeat_every, end_after)
        except NoOptionsAvailable:
            raise NoLiveChannels()
        self.last_vote_at = datetime.datetime.now()
        self.track_vote_start()
        return self.vote

    def end_vote(self):
        '''
        Ends the current vote and tracks final state
        '''
        self.track_vote_end()
        self.vote = None

    @property
    def is_vote_completed(self):
        '''
        Returns True if the vote is finished
        '''
        if not self.vote:
            raise NoVoteInProgress()
        return (self.last_vote_at + self.vote.end_after <
                datetime.datetime.now())

    @property
    def tracking_fields(self):
        fields = {
            "community": self.name,
            "promoted_channel": self.promoted_channel
        }
        options = dict([
            (("option_%d" % opt_num), channel)
            for opt_num, (channel, _) in enumerate(self.vote.scores_by_channel)
        ])
        fields.update(options)
        scores = dict([
            (("option_%d_score" % opt_num), score)
            for opt_num, (_, score) in enumerate(self.vote.scores_by_channel)
        ])
        fields.update(scores)
        return fields

    def track_vote_start(self):
        track("community_vote_start", self.tracking_fields)

    def track_vote_end(self):
        track("community_vote_end", self.tracking_fields)
