from __future__ import print_function
from itertools import islice
from operator import attrgetter
import math
import random
from datetime import datetime, timedelta
from communitybot.settings import SETTINGS
import communitybot.logger as logger
from communitybot.clients.twitch import TwitchApi, TwitchApiError
from communitybot.util import thirds
from communitybot.clients.mission_control import (
    MissionControlClient,
    MissionControlClientError
)


class NoOptionsAvailable(Exception):
    def __str__(self):
        return repr("No Options Available")


class Option(object):
    '''
    Vote option that contains requried properties for displaying
    and tracking during a vote. Custom `resource` property used to
    store a dynamic object that is being voted upon.
    '''
    def __init__(self, emote, id, title, resource=None, score=0):
        self.emote = emote
        self.id = id
        self.title = title
        self.score = score
        self.resource = resource

    def __str__(self):
        return '%s with %s votes' % (self.title, self.score)


class ChannelOption(Option):
    '''
    Custom Option that contains Channel specific helpers and display
    '''

    def __init__(self, *args, **kwargs):
        self.is_rerun = kwargs.pop('is_rerun', False)
        super(ChannelOption, self).__init__(*args, **kwargs)

    @property
    def url(self):
        return self.resource.url

    def __str__(self):
        jw = self.is_rerun and 'for' or 'with'
        return '%s %s %s %s votes' % (self.title, self.url, jw, self.score)


class Vote(object):
    '''
    Represents a single instance that will
    '''

    def __init__(self, repeat_every=None, end_after=None):
        if repeat_every is None:
            repeat_every = timedelta(seconds=SETTINGS['vote_repeat_every'])
        if end_after is None:
            end_after = timedelta(seconds=SETTINGS['vote_end_after'])
        self.repeat_every = repeat_every
        self.end_after = end_after
        self.voters = set()
        self.last_vote_announcement = None
        self.options = []

    def vote(self, voter, msg):
        '''
        Place a vote on behalf of the voter, if they have not already voted
        '''
        if voter in self.voters:
            return
        option = self.option_from_msg(msg)
        if option:
            option.score += 1
            self.voters.add(voter)

    def is_valid_vote(self, msg):
        '''
        Checks if a message contains a valid vote. A valid vote starts with
        an option's emote or channel name
        '''
        return self.option_from_msg(msg) is not None

    def option_from_msg(self, msg):
        '''
        Return the emote that a provided message contains
        '''
        m = msg.lower()
        for option in self.options:
            name = option.title.lower()
            emote = option.emote.lower()
            if m.startswith(emote) or msg.startswith(name):
                return option
        return None

    @property
    def scores_by_channel(self):
        '''
        Return the current scores of each channel
        '''
        return sorted([(o.title, o.score) for o in self.options])

    @property
    def winner(self):
        '''
        Return the current winner
        '''
        return max(self.options, key=attrgetter('score'))

    @property
    def is_time_to_announce(self):
        '''
        Return True when the vote should announce
        '''
        if self.last_vote_announcement is None:
            return True
        next_announcement = self.last_vote_announcement + self.repeat_every
        return next_announcement < datetime.now()

    def update_last_vote_announcement(self):
        '''
        Set the last announced time to the current time
        '''
        self.last_vote_announcement = datetime.now()


class StreamVote(Vote):

    def __init__(self, channel, emotes, **kwargs):
        super(StreamVote, self).__init__()
        self.set_options_from_stream(channel, emotes)

    def set_options_from_stream(self, channel, emotes):
        '''
        Randomly select options from provided stream channel
        '''
        host = SETTINGS['mission_control_api']
        client = MissionControlClient(host=host, channel=channel)
        videos, _ = client.get_videos(limit=len(emotes), random=True)
        if len(videos) == 0:
            raise NoOptionsAvailable()
        ve = enumerate(videos)
        self.options = [Option(emotes[i], v.id, v.title, v) for i, v in ve]


class CommunityVote(Vote):

    def __init__(self, community_name, emotes, **kwargs):
        super(CommunityVote, self).__init__()
        self.set_options_from_community(community_name, emotes)

    def set_options_from_community(self, community_name, emotes):
        '''
        Randomly select options from Community
        '''
        try:
            community = TwitchApi.get_community(community_name)
            streams, _ = TwitchApi.get_streams_by_community(community_name)
        except TwitchApiError:
            raise NoOptionsAvailable()
        promoted = community.promoted_channel
        channels = [s.channel for s in streams]
        current = next((c for c in channels if c.name == promoted), None)
        pool = thirds([o for o in channels if o != current])
        choices = [random.choice(c) for c in pool]
        if len(choices) == 0:
            raise NoOptionsAvailable()
        self.options = [ChannelOption(emotes[i], c.name, c.display_name, c)
                        for i, c in enumerate(choices)]
        if current is not None:
            emote = emotes[len(self.options)]
            rerun = ChannelOption(emote, current.name, current.display_name,
                                  current, is_rerun=True)
            self.options.append(rerun)

    @property
    def channels(self):
        '''
        Return each channel in the current vote
        '''
        return [c.id for c in self.options]
