from __future__ import print_function
import argparse
import logger
import os
import sys
import subprocess
import itertools
import ffmpeg
import datetime
import irc.client
from irc.bot import SingleServerIRCBot
from video import Video
from dreamer import write_next_queue
from util import throttle
from settings import SETTINGS

SERVER = 'irc.twitch.tv'
PORT = 6667
ADMINS = ['jimmyhillis', 'jwahba', 'bill', 'monkeyonstrike']


class Vote(object):
    '''
    Run a user vote. Tracks each option and each user that has voted
    allowing you to determine the winner. Each user can only vote once,
    and they must vote for an existing option
    '''

    class VoteError(Exception):
        '''
        A Vote specific error
        '''

    class InvalidOptionError(VoteError):
        '''
        Option provided was not part of this vote
        '''

    class MultipleVotesError(VoteError):
        '''
        User has already voted
        '''

    def __init__(self, options):
        self.votes = dict([(i, []) for i in options])

    @property
    def options(self):
        '''
        Return the options available to vote on
        '''
        return self.votes.keys()

    @property
    def winner(self):
        '''
        Return the option which currently has the most votes
        '''
        return max(self.votes.iterkeys(),
                   key=(lambda key: len(self.votes[key])))

    @property
    def scores(self):
        return [(key, len(self.votes[key])) for key in self.votes.iterkeys()]

    @property
    def users(self):
        '''
        Return a list of each username that has voted
        '''
        return list(itertools.chain(*self.votes.values()))

    def option_from_emote(self, emote):
        '''
        Return the option by emote. Each option will be designated
        a random emote and can be voted for that way
        '''
        for index, emote_option in enumerate(SETTINGS['bot']['emotes']):
            if emote.startswith(emote_option):
                return self.options[index]

    def emote_from_option(self, option):
        '''
        Return the emote by option.
        '''
        if option in self.options:
            index = self.options.index(option)
            return SETTINGS['bot']['emotes'][index]

    def option_from_number(self, index):
        '''
        Return the option by number. Each option will be designated
        1, 2, 3, 4 in counting order for quickly voting without the name
        '''
        if index <= len(self.options):
            return self.options[index - 1]
        return None

    def has_voted(self, user):
        '''
        Return True if user has voted in this vote
        '''
        return user in self.users

    def vote(self, option, user, weight=1):
        '''
        Vote for a provided episode with weight votes
        '''
        if option not in self.options:
            raise Vote.InvalidOptionError
        if self.has_voted(user):
            raise Vote.MultipleVotesError
        self.votes[option].append(user)


class TwitchBot(SingleServerIRCBot):
    '''
    Generalized Twitch bot for interacting with a channel via
    TMI/IRC Twitch Chat
    '''

    def __init__(self, nickname=None, password=None, channels=[],
                 server=SERVER, port=PORT, verbose=False):
        server_list = [(server, port, password)]
        super(TwitchBot, self).__init__(server_list, nickname, nickname)
        self._channels = channels
        self._mods = dict([(i, []) for i in self._channels])
        self._tracking = []
        self.verbose = verbose

    def on_welcome(self, conn, event):
        '''
        On server connection, join requested channels
        '''
        conn.send_raw('CAP REQ :twitch.tv/membership')
        conn.send_raw('CAP REQ :twitch.tv/commands')
        for channel in self._channels:
            conn.join(channel)

    # Extremely helpful debugging tool when tracking IRC commands
    # def on_all_raw_messages(self, conn, event):
    #     print(event)

    def on_pubnotice(self, conn, event):
        '''
        Watch for MOD status on pubnotice
        '''
        channel = event.target
        if event.arguments[0].startswith('The moderators of this room are: '):
            mods = [m.strip() for m in event.arguments[0][33:].split(',')]
            self._mods[channel] = mods

    def on_join(self, conn, event):
        '''
        Track status of each joined channel
        '''
        channel = event.target
        if channel not in self._tracking:
            self._tracking.append(channel)
            self._request_mods(conn, channel)
            self.reactor.execute_every(30, self._request_mods, (conn, channel))

    def _request_mods(self, conn, channel):
        conn.privmsg(channel, '/mods')

    def mods(self, channel):
        '''
        Return the mods within the provided target
        '''
        return self._mods[channel]

    def is_owner(self, channel, user):
        '''
        Return True when if the user is the owner of the channel
        '''
        return channel[1:] == user or user in ADMINS

    def is_mod(self, channel, user):
        '''
        Return True if the user is a mod of the channel
        '''
        return user in self.mods(channel)

    def _get_user_msg_from_irc(self, event):
        '''
        Return a tuple of (username, message) from an IRC event
        '''
        return event.source.split('!')[0], event.target, event.arguments[0]

    def say(self, conn, channel, msg):
        if self.verbose:
            self._log(conn, channel, msg)
        conn.privmsg(channel, '/me %s' % msg)

    def _log(self, conn, channel, msg):
        '''
        Testing proxy message call to test into my own channel only
        '''
        print('[%s]: %s' % (channel, msg))


class RequestBot(TwitchBot):

    COMMANDS = [
        '!whatsplaying',
        '!whatsnext'
    ]

    MOD_COMMANDS = []

    OWNER_COMMANDS = [
        '!request',
        '!requestover',
        '!requesttime',
        '!requesttimeover',
        '!startstream',
        '!stopstream'
    ]

    START_BUFFER = datetime.timedelta(seconds=60)
    END_BUFFER = datetime.timedelta(minutes=2)
    MAX_VOTE_DURATION = datetime.timedelta(minutes=5)

    def __init__(self, **kwargs):
        super(RequestBot, self).__init__(**kwargs)
        self._votes = self.votes = dict([(i, None) for i in self._channels])
        self._request_times = dict([(i, ()) for i in self._channels])
        self._vote_scheduled = dict([(i, False) for i in self._channels])

    def _parse_command(self, msg, commands):
        '''
        Parse a message searching for a known (and supported) command
        '''
        parts = msg.split(' ')
        if parts and msg.startswith('!') and parts[0] in commands:
            return parts[0], parts[1:]
        return None, None

    def is_vote(self, msg):
        '''
        Returns True if the message is a valid vote
        '''
        return False

    def set_vote(self, channel, vote):
        self._votes[channel] = vote
        return self.vote(channel)

    def vote(self, channel):
        '''
        Return the vote for a specific channel
        '''
        return self._votes[channel]

    def on_pubmsg(self, conn, event):
        '''
        Track each message and parse for commands
        '''
        user, channel, msg = self._get_user_msg_from_irc(event)
        commands = self.COMMANDS[:]
        if self.is_mod(channel, user):
            commands += self.MOD_COMMANDS[:]
        if self.is_owner(channel, user):
            commands += self.OWNER_COMMANDS[:]
        command, args = self._parse_command(msg, commands)
        if command:
            return self.do_command(conn, channel, command, args)
        vote = self.vote(channel)
        if vote:
            try:
                option = vote.option_from_emote(event.arguments[0])
                vote.vote(option, user)
                self._log(conn, channel, '%s voted for %s' % (user, option))
            except ValueError:
                pass
            except Vote.VoteError, Vote.MultipleVotesError:
                pass

    def do_command(self, conn, channel, command, args=()):
        '''
        Run the provided command against the provided connection
        and channel
        '''
        if command == '!whatsplaying':
            self.whatsplaying(conn, channel)
        elif command == '!whatsnext':
            self.whatsnext(conn, channel)
        elif command == '!request':
            self.start_vote(conn, channel, *args[:2])
        elif command == '!requestover':
            self.end_vote(conn, channel)
        elif command == '!requesttime':
            self.start_request_time(conn, channel)
        elif command == '!requesttimeover':
            self.stop_request_time(conn, channel)
        elif command == '!startstream':
            self.start_stream(conn, channel)
        elif command == '!stopstream':
            self.stop_stream(conn, channel)

    @throttle(seconds=10)
    def whatsplaying(self, conn, channel):
        '''
        Send the currently playing episode
        '''
        current_video = ffmpeg.current_video()
        if current_video:
            self.say(conn, channel, 'Now playing "%s"' %
                     current_video.title)
        else:
            self.say(conn, channel, 'Nothing, right now')

    @throttle(seconds=10)
    def whatsnext(self, conn, channel):
        '''
        Send the next episode
        '''
        next_video = ffmpeg.next_video()
        if self._vote_scheduled[channel]:
            self.say(conn, channel, 'I will let the people decide!')
        elif next_video:
            self.say(conn, channel, 'Will play "%s" next'
                     % next_video.title)
        else:
            self.say(conn, channel, 'Still thinking about it')

    def start_vote(self, conn, channel, repeat_every=30,
                   end_after=MAX_VOTE_DURATION.seconds):
        '''
        Start a new vote with random episodes
        '''
        try:
            repeat_every = int(repeat_every)
            if end_after is not None:
                end_after = int(end_after)
        except ValueError:
            self.say(conn, channel, 'Sorry? Don\'t think so!')
        if self.vote(channel):
            logger.log('Attempted to start a vote while one was already going')
            return
        vote = self.set_vote(channel, Vote(Video.random(count=3)))

        def say_vote():
            '''
            Send vote options to chat
            '''
            if self.vote(channel) == vote:
                self.say(conn, channel, 'TIME TO VOTE!')
                for option, score in vote.scores:
                    self.say(conn, channel, '%s for "%s" vote count: %s' % (
                             vote.emote_from_option(option),
                             option.title,
                             score))
            self.reactor.execute_delayed(repeat_every, say_vote)

        say_vote()
        if end_after is not None:
            args = (conn, channel)
            self.reactor.execute_delayed(end_after, self.end_vote, args)

    def end_vote(self, conn, channel):
        '''
        End the current vote and save the next queue
        '''
        vote = self.vote(channel)
        if not vote:
            logger.log('Attempted to end a vote without one going')
            return
        winner = vote.winner
        write_next_queue(videos=[winner])
        vote_count = dict(vote.scores)[winner]
        self._log(conn, channel, str(vote.votes))
        self.say(
            conn,
            channel,
            'THE VOTE IS DONE! The winner is "%s" with %s votes' %
            (winner.title, vote_count)
        )
        self._vote_scheduled[channel] = False
        self.set_vote(channel, None)

    def start_request_time(self, conn, channel):
        '''
        Begins request time
        '''
        if self._request_times[channel]:
            logger.log('Attempted to start request time while one was going')
            return
        self.say(conn, channel, 'Do you know what time it is? REQUEST TIME!')
        self._request_times[channel] = ffmpeg.unfinished_video()
        self.reactor.execute_delayed(1, self.run_request_time, (channel,))

    def run_request_time(self, channel):
        '''
        Run loop that recurses unless request time is over
        '''
        if not self._request_times[channel]:
            return
        current_queue = ffmpeg.current_queue()
        if current_queue:
            last_video_in_queue = current_queue.videos[-1]
            if self.did_video_change(channel):
                video, timestamp = self._request_times[channel]
                if last_video_in_queue == video:
                    self.schedule_request(video, timestamp, channel)
        self.reactor.execute_delayed(1, self.run_request_time, (channel,))

    def did_video_change(self, channel):
        '''
        Checks if the video has changed since the last time the methods was
        called
        '''
        last_play_event = ffmpeg.unfinished_video()
        if self._request_times[channel] != last_play_event:
            self._request_times[channel] = last_play_event
            return True
        else:
            return False

    def schedule_request(self, video, timestamp, channel):
        '''
        Schedule a request during the current video for the next video if
        possible
        '''
        self._vote_scheduled[channel] = True
        if video.duration < datetime.timedelta(minutes=3):
            return
        votable_time = video.duration - self.END_BUFFER - self.START_BUFFER
        request_end = video.duration - self.END_BUFFER
        vote_duration = min(self.MAX_VOTE_DURATION, votable_time)
        request_start = request_end - vote_duration
        self.reactor.execute_delayed(
            request_start.seconds,
            self.start_vote,
            (self.connection, channel, 30, vote_duration.seconds)
        )

    def stop_request_time(self, conn, channel):
        '''
        Ends request time but doesn't stop the current vote
        '''
        self.say(conn, channel, "That was fun! I'm taking a break")
        self._request_times[channel] = ()

    def start_stream(self, conn, channel):
        self.say(conn, channel, 'Buckle up, we\'re getting started!')
        cmd = 'sudo supervisorctl start dreamer-stream'
        subprocess.check_output(cmd, shell=True)

    def stop_stream(self, conn, channel):
        self.say(conn, channel, 'Thanks for watching, see you soon!')
        cmd = 'sudo supervisorctl stop dreamer-stream'
        subprocess.check_output(cmd, shell=True)

if __name__ == '__main__':
    logger.basicConfig(
        SETTINGS['environment'],
        rollbar=SETTINGS['rollbar'],
        slack=SETTINGS['slack']
    )
    try:
        bot = RequestBot(
            channels=SETTINGS['bot']['irc_channels'],
            nickname=SETTINGS['bot']['nickname'],
            password=SETTINGS['bot']['oauth_token'],
            server=SETTINGS['bot']['server'],
            port=SETTINGS['bot']['port'],
            verbose=SETTINGS['verbose']
        )
        bot.start()
    except Exception:
        logger.exception(sys.exc_info())
