#!/usr/bin/env python

'''
Admin server for managing queues for 24/7 streaming to Twitch.tv
'''

from __future__ import print_function
import argparse
import logger
import os
import math
import tornado.ioloop
import tornado.web
import ffmpeg
import sys
import datetime
from dateutil import tz
from show import Show
from video import Video
from queue import Queue
from collections import defaultdict
from settings import SETTINGS


def format_timestamp(timestamp):
    if timestamp:
        return timestamp.strftime("%-d %-b %-H:%M")


def format_timedelta(timedelta):
    '''
    Return time formatted as HH:MM:SS
    '''
    seconds = timedelta.total_seconds()
    hours = math.floor(seconds / 3600)
    seconds -= hours * 3600
    minutes = math.floor(seconds / 60)
    seconds -= minutes * 60
    return '%dh:%dm:%ds' % (hours, minutes, seconds)


def format_episode(video):
    '''
    Return episode formatted as `EPISODE [HH:MM:SS]`
    '''
    return '%s [%s]' % (video.path, format_timedelta(video.duration))


def format_show(show):
    '''
    Return episode formatted as `SHOW [HH:MM:SS]`
    '''
    return '%s [%s]' % (show.path, format_timedelta(show.duration))


def write_next_queue(videos=None):
    '''
    Write the next queue file to contain provided videos, or a single random
    video if none are provided.
    '''
    if not videos:
        videos = Video.random(count=1)
    queue = ffmpeg.next_queue()
    queue.update(videos=videos)
    logger.log('New queue - \'%s\'' % queue.path)
    return queue


def move_old_queues():
    '''
    Archive all old queues from the working directory to keep track
    of what videos have been played.
    '''
    queues = Queue.all()
    for queue in queues[:queues.index(ffmpeg.current_queue())]:
        logger.log('Archive queue - \'%s\'' % queue.path)
        queue.archive()


FFMPEG_FAILURE_HAS_BEEN_LOGGED = False


def stay_sane():
    '''
    Loop callback will ensure the current queues are in order,
    clean up and log any changes required.
    '''
    global FFMPEG_FAILURE_HAS_BEEN_LOGGED
    if ffmpeg.next_queue().is_persisted is False:
        write_next_queue()
    if ffmpeg.is_streaming():
        FFMPEG_FAILURE_HAS_BEEN_LOGGED = False
        move_old_queues()
    elif FFMPEG_FAILURE_HAS_BEEN_LOGGED is False:
        logger.log('ERROR: Cannot find FFMPEG.', level=logger.ERROR)
        FFMPEG_FAILURE_HAS_BEEN_LOGGED = True


def queue_as_html(title, queue, highlight=None):
    '''
    Return an HTML formatted list of provided queue
    '''
    if not queue:
        return None
    video_markup = []
    for video in queue.videos:
        if highlight and video == highlight:
            video_markup.append('<span style="color:red">%s</span>' %
                                format_episode(video))
        else:
            video_markup.append(format_episode(video))
    return '<h2>%s [%s]</h2>%s' % (title, format_timedelta(queue.duration()),
                                   '<br />'.join(video_markup))


def all_videos_html():
    '''
    Return an HTML list of all available video files
    '''
    mp4s = Video.all(sort=True)
    mp4s = ['%s - <a href="/append?mp4=%s">append</a> | '
            '<a href="/replace?mp4=%s">replace</a> | '
            '<a href="/interleave?mp4=%s">interleave</a>' %
            (format_episode(x), x.path, x.path, x.path)
            for x in mp4s]
    shows = Show.all()
    shows = ['%s - <a href="/append?show=%s">append</a> | '
             '<a href="/replace?show=%s">replace</a>' %
             (format_show(show), show.path, show.path)
             for show in shows]
    return ('<h3>Full shows:</h3>' + '<br>'.join(shows) +
            '<h3>All individual episodes</h3>' + '<br>'.join(mp4s))


class MainHandler(tornado.web.RequestHandler):

    def get(self):
        current_queue = ffmpeg.current_queue()
        next_queue = ffmpeg.next_queue()
        current_video = ffmpeg.current_video()
        current_html = queue_as_html('Current (cannot be modified)',
                                     current_queue, current_video)
        video, timestamp = ffmpeg.unfinished_video()
        if current_queue:
            next_start = current_queue.estimated_finish_time(
                video,
                timestamp,
                timezone=tz.gettz('America/Los_Angeles')
            )
        else:
            next_start = None
        next_title = ('Next queue (starts %s)' %
                      (format_timestamp(next_start) or 'TBA'))
        next_html = queue_as_html(next_title, next_queue)
        videos_html = all_videos_html()
        self.write("""<!DOCTYPE html>
            <html style="margin:20px">
            <head>
            <script src="static/jquery-2.2.1.min.js"></script>
            <link rel="stylesheet" href="static/css/bootstrap.min.css">
            <script src="static/js/bootstrap.min.js"></script>
            </head>
            <body>
            %s
            %s
            <h2>Available shows:</h2>
            %s
            </body>
            </html>
            """ % (current_html or '', next_html or '', videos_html or ''))


class ReplaceHandler(tornado.web.RequestHandler):

    def get(self):
        if self.get_argument('mp4', False):
            mp4 = self.get_argument('mp4')
            write_next_queue([Video(mp4)])
        elif self.get_argument('show', False):
            show = self.get_argument('show')
            mp4s = [x for x in Video.all(sort=True) if x.path.startswith(show)]
            write_next_queue(mp4s)
        self.redirect('/')


class AppendHandler(tornado.web.RequestHandler):

    def get(self):
        if self.get_argument('mp4', False):
            mp4 = self.get_argument('mp4')
            write_next_queue(ffmpeg.next_queue().videos + [Video(mp4)])
        elif self.get_argument('show', False):
            show = self.get_argument('show')
            mp4s = [x for x in Video.all(sort=True) if x.path.startswith(show)]
            write_next_queue(ffmpeg.next_queue().videos + mp4s)
        self.redirect('/')


class InterleaveHandler(tornado.web.RequestHandler):

    def interleave(self, videos, x):
        ret = [x]
        for a in videos:
            ret = ret + [a, x]
        return ret

    def get(self):
        if self.get_argument('mp4', False):
            mp4 = self.get_argument('mp4')
            videos = self.interleave(ffmpeg.next_queue().videos, Video(mp4))
            write_next_queue(videos)
        self.redirect('/')

if __name__ == '__main__':
    logger.basicConfig(SETTINGS['environment'],
                       rollbar=SETTINGS['rollbar'],
                       slack=SETTINGS['slack'])
    try:
        settings = {
            'static_path': os.path.join(os.path.dirname(__file__), 'static'),
        }
        app = tornado.web.Application([
            (r'/', MainHandler),
            (r'/replace', ReplaceHandler),
            (r'/append', AppendHandler),
            (r'/interleave', InterleaveHandler),
        ], **settings)
        app.listen(SETTINGS['dreamer']['port'])
        task = tornado.ioloop.PeriodicCallback(stay_sane, 1000)
        task.start()
        tornado.ioloop.IOLoop.current().start()
    except Exception:
        logger.exception(sys.exc_info())
