'''
Start, and maintain, a 24/7 stream using `ffmpeg`
'''

from __future__ import print_function
import os
import re
import logger
import subprocess
import datetime
import requests
import json
import sys
from video import Video
from queue import Queue


class NotStreamingError(Exception):
    '''
    Instance of ffmpeg is not currently streaming
    '''


def pid(pid=None):
    '''
    Return or set the pid for the currently running ffmpeg instance.
    May raise NotStreamingError exception on get.
    '''
    if pid is not None:
        with open('FFMPEG_PID', 'w+', 0) as f:
            f.write(str(pid))
        return
    with open('FFMPEG_PID', 'r') as f:
        try:
            pid = int(f.read().strip())
            os.kill(pid, 0)
            return pid
        except (OSError, ValueError):
            raise NotStreamingError


def is_streaming():
    '''
    Return true if there is an instance of ffmpeg currently streaming
    with a PID matching the stored FFMPEG_PID file
    '''
    try:
        pid()
        return True
    except NotStreamingError:
        pass
    return False


def current_queue():
    '''
    Return the queue currently being streamed.
    '''
    try:
        # ffmpeg holds *every* queue file open,
        # even ones it should be finished with, forever.  Ugh.
        lsof = subprocess.check_output('lsof -p %s -Fn' % pid(), shell=True)
        files = sorted([f for f in lsof.split('\n') if f.endswith('.txt')],
                       key=os.path.basename)
        return Queue(os.path.basename(files[-1][1:]))
    except NotStreamingError:
        pass
    return None


def next_queue():
    '''
    Return the queue that ffmpeg will stream next. This will work with
    ffmpeg streaming or not streaming.
    '''
    current = current_queue()
    if current:
        return current.next_queue
    else:
        queues = Queue.all()
        if queues:
            return queues[0]
    return Queue.find_or_create_by_index(1)


def current_video():
    '''
    Return the video currently being streamed.
    TODO: Try and work this work without tracking the process as it's messy
    and doesn't promise that anything will work
    '''
    try:
        lsof = subprocess.check_output('lsof -p %s -Fn' % pid(), shell=True)
        basename = os.path.basename
        videos = [Video(f) for f in lsof.split('\n') if f.endswith('.mp4')]
        if not videos:
            logger.log('FFMPEG is not playing a video right now.',
                       level=logger.WARNING)
            return None
        if len(videos) > 1:
            logger.log('FFMPEG has 2 videos [%s]' %
                       '|'.join([v.path for v in videos]),
                       level=logger.WARNING)
        return videos[0]
    except NotStreamingError:
        pass
    except subprocess.CalledProcessError:
        pass
    return None


def next_video():
    '''
    Return the next video that will be played
    '''
    current = current_video()
    queue = current_queue()
    if queue is None:
        return None
    videos = queue.videos
    if current in videos:
        next_index = videos.index(current) + 1
        if len(videos) > next_index:
            return videos[next_index]
        next_queue_videos = next_queue().videos
        if next_queue_videos:
            return next_queue_videos[0]
    return None


def unfinished_video(video=None):
    '''
    Return or set an unfinished video being streamed by ffmpeg
    '''
    if video is not None:
        with open('FFMPEG_PLAYING', 'w+', 0) as f:
            f.seek(0)
            f.write(video.path or '')
            f.write("\t")
            timestamp = datetime.datetime.utcnow()
            f.write(str(timestamp))
            f.truncate()
        return (video, timestamp)
    try:
        with open('FFMPEG_PLAYING', 'r') as f:
            contents = f.readline().strip().split("\t")
            while len(contents) < 2:
                contents.append(None)
            file_path, time_started = contents
            video = Video(file_path) if file_path else None
            if time_started:
                timestamp = datetime.datetime.strptime(
                    time_started,
                    "%Y-%m-%d %H:%M:%S.%f"
                )
            else:
                timestamp = None
            return (video, timestamp)

    except IOError:
        return (None, None)


def change_stream_title(video, channel, oauth_token):
    '''
    Change stream title to include the video's show
    '''
    stream_title = video.show.title + " on #food"
    headers = {
        'Authorization': 'OAuth ' + oauth_token,
        'Accept': 'application/vnd.twitchtv.v2+json',
        "Content-Type": 'application/json'
    }
    data = {
        "channel": {
            "status": stream_title
        }
    }
    url = "https://api.twitch.tv/kraken/channels/" + channel
    resp = requests.put(url, headers=headers, data=json.dumps(data))
    return resp


def stream(queue, stream_key, channel, oauth_token):
    '''
    Launch ffmpeg and return the process

    bash example
    ------------
        ffmpeg -re -auto_convert 1 -f concat -i $queue -c copy \
            -f flv "rtmp://live.twitch.tv/app/$STREAM_KEY"
    '''
    command = [
        'ffmpeg',
        '-re',
        '-auto_convert', '1',
        '-f', 'concat',
        '-i', queue.sanitized_path(),
        '-c', 'copy',
        '-f', 'flv',
        'rtmp://live.twitch.tv/app/%s' % stream_key
    ]
    logger.log('Attempting to call `%s`' % subprocess.list2cmdline(command))
    process = subprocess.Popen(command,
                               stdout=subprocess.PIPE,
                               stderr=subprocess.STDOUT)
    pid(process.pid)
    last_video = None
    while True:
        nextline = process.stdout.readline()
        log_errors(nextline)
        if nextline == '' and process.poll() is not None:
            break
        current = current_video()
        if current != last_video:
            last_video = current
            if last_video is not None:
                video, _ = unfinished_video(last_video)
                if oauth_token is not None:
                    try:
                        change_stream_title(video, channel, oauth_token)
                    except Exception:
                        logger.exception(sys.exc_info())
                    logger.log('Starting video - \'%s\'' % last_video.title,
                               level=logger.DEBUG)
                    logger.slack('#creative-stream',
                                 'Now playing %s' % last_video.title)


def log_errors(message):
    '''
    Check the output of ffmpeg for any errors that be displayed and caught.
    We can leverage this information to cleanup and restart better. This
    method will use stderr and logging from ffmpeg to determine why
    it's not longer running to diagnose the problem as well as possible.
    '''
    # Networking issues
    if ('Failed to resolve hostname' in message or
            'Error writing trailer' in message or
            'Conversion failed' in message):
        logger.log('NETWORKING. Hostname resolution\n-%s' % message,
                   level=logger.ERROR)
    if 'Cannot open connection' in message:
        logger.log('NETWORKING. Connection to API\n-%s' % message,
                   level=logger.ERROR)
    # Content issues
    if 'Impossible to open' in message:
        # Error is a file issue, we can deal with a QUEUE issue but
        # a video issue is super strange, who touched it?
        res = re.search('.* Impossible to open \'(.*)\'', message)
        if res is not None:
            missing_file = os.path.basename(res.group(1))
            if (missing_file.endswith('.mp4')):
                logger.log('Missing video file %s' % missing_file,
                           level=logger.ERROR)
            elif (missing_file.startswith('list_') and
                    missing_file.endswith('.txt')):
                logger.log('Missing queue file %s' % missing_file,
                           level=logger.ERROR)
