from __future__ import print_function
import rollbar as _rollbar
from slacker import Slacker
import sys

IS_DEBUG = False
ROLLBAR_KEY = None
SLACK_KEY = None

CRITICAL = 'critical'
ERROR = 'error'
WARNING = 'warning'
INFO = 'info'
DEBUG = 'debug'
IGNORED = 'ignored'

LEVELS = {
    CRITICAL: CRITICAL,
    ERROR: ERROR,
    WARNING: WARNING,
    INFO: INFO,
    DEBUG: DEBUG,
    IGNORED: IGNORED
}

ERROR_LEVELS = [CRITICAL, ERROR]


def set_debug_mode(debug):
    '''
    Set global debug mode for logger. If DEBUG is True then no
    logging will be sent to external sources. Good for local testing.
    '''
    global IS_DEBUG
    IS_DEBUG = debug


def set_rollbar_key(key, environment):
    global ROLLBAR_KEY
    ROLLBAR_KEY = key
    _rollbar.init(ROLLBAR_KEY, environment)


def set_slack_key(key):
    '''
    Set a global slack key for sending messages through a Slackbot
    '''
    global SLACK_KEY
    SLACK_KEY = key


def log(message, level=INFO):
    '''
    Log provided message to the right tool based on configuration
    and the level provided. Supports INFO, WARNING and ERROR.
    '''
    output = (level in ERROR_LEVELS and sys.stderr) or sys.stdout
    print('%s: %s' % (level, message), file=output)
    rollbar(message, level)


def exception(info):
    '''
    Log application exception
    '''
    import traceback
    print('EXCEPTION: %s' % traceback.format_exc(), file=sys.stderr)
    _rollbar.report_exc_info(info)


def rollbar(message, level):
    if ROLLBAR_KEY:
        try:
            _rollbar.report_message(message, level)
        except Exception:
            print('ERROR: Rollbar failure.', file=sys.stderr)


def slack(channel, message):
    '''
    Log message to Slack channel
    '''
    if IS_DEBUG is False and SLACK_KEY is not None:
        slack = Slacker(SLACK_KEY)
        if not channel.startswith('#'):
            channel = '#%s' % channel
        try:
            slack.chat.post_message(channel, message, as_user=True)
        except Exception:
            print('ERROR: Slack failure.', file=sys.stderr)


def basicConfig(environment, rollbar=None, slack=None):
    '''
    Configure external logger
    '''
    if environment == 'development':
        set_debug_mode(True)
    if rollbar:
        set_rollbar_key(rollbar, environment)
    if slack:
        set_slack_key(slack)
