import datetime
import os
import s3
from dateutil import tz
from video import Video
import operator

QUEUE_PATH = './content'


class InvalidQueueFile(Exception):
    '''
    File is not a valid queue
    '''


class Queue(object):

    def __init__(self, path, is_archived=False):
        self.path = path
        self.is_archived = is_archived

    def __eq__(self, other):
        return self.path == other.path

    @staticmethod
    def set_queue_path(path):
        '''
        Update global path for queue files on running instance
        '''
        global QUEUE_PATH
        QUEUE_PATH = os.path.realpath(path)

    @staticmethod
    def is_queue_path(path):
        '''
        Return True when path is a valid queue path
        '''
        return path.endswith('.txt')

    @staticmethod
    def all():
        '''
        Return a list of all active queue files
        '''
        files = os.listdir(QUEUE_PATH)
        queues = [Queue(f) for f in files if Queue.is_queue_path(f)]
        return sorted(queues, key=lambda queue: queue.path)

    @staticmethod
    def archived():
        files = os.listdir(os.path.join(QUEUE_PATH, 'old_lists'))
        queues = [Queue(f, is_archived=True) for f in files
                  if Queue.is_queue_path(f)]
        return sorted(queues, key=lambda queue: queue.path)

    @staticmethod
    def first():
        '''
        Return a path to the first available queue file
        '''
        queues = Queue.all()
        if queues:
            return queues[0]
        return None

    @staticmethod
    def last():
        '''
        Return a path to the last available queue file
        '''
        queues = Queue.all()
        if queues:
            return queues[-1]
        return None

    @staticmethod
    def write_queue(queue_path, videos, queue_after, unfinished=False):
        '''
        Write a new queue file with id as the key and videos list as
        provided content.
        TODO: Validate that each video exists
        TODO: kill this method and  use #update path instead
        '''
        video_list = '\n'.join(["file '%s'" % video.path for video in videos])
        path = queue_path
        with open(path, 'w') as f:
            f.write("ffconcat version 1.0\n# Written %s\n%s\n" % (
                datetime.datetime.now(), video_list))
            if queue_after:
                f.write("file '%s'\n" %
                        os.path.basename(queue_after.path))
        return Queue(os.path.basename(path))

    @staticmethod
    def find_or_create_by_index(index):
        '''
        Return standard filename for a queue based on the filename
        '''
        return Queue('list_%010d.txt' % index)

    @property
    def next_queue(self):
        return Queue.find_or_create_by_index(self.index + 1)

    @property
    def index(self):
        '''
        Return the index of a queue from it's filename
        '''
        try:
            return int(self.path[5:15])
        except ValueError:
            return None

    @property
    def is_persisted(self):
        '''
        Return True when provided queue exists and is valid
        '''
        return os.path.isfile(self.sanitized_path())

    @property
    def videos(self):
        '''
        Load an ffmpeg queue file into memory, and return the video files in a
        list
        TODO: Make reusable
        '''
        try:
            with open(self.sanitized_path()) as f:
                files = [x for x in f.read().split('\n') if (
                    x.startswith('file ') and '.mp4' in x)]
                return [Video(x.split("'")[1]) for x in files]
        except:
            return []

    def insert_before(self, videos):
        '''
        Inserts a queue before the next queue
        This has to bump all the other queues introducing a low risk race
        condition
        '''
        queues = Queue.all()[queues.index(self):] + [Queue.last().next_queue]
        _videos = [videos] + [q.videos for q in queues]
        for queue, queue_videos in zip(queues, _videos):
            queue.update(videos=queue_videos)
        return self

    def estimated_finish_time(self, video, offset, timezone=tz.gettz('UTC')):
        '''
        Returns how long is left given a video in the queue and an offset as a
        timedelta
        '''
        if not video or not offset:
            return None
        videos = self.videos
        videos = videos[videos.index(video):]
        time_left = reduce(
            operator.add,
            [v.duration for v in videos],
            datetime.timedelta(hours=0)
        )
        finish_time = offset + time_left
        finish_time_utc = finish_time.replace(tzinfo=tz.gettz('UTC'))
        return finish_time_utc.astimezone(timezone)

    def update(self, videos, queue_after=None):
        '''
        Updates the current queue with the videos and the next queue
        '''
        queue_after = queue_after or self.next_queue
        video_list = '\n'.join(["file '%s'" % video.path for video in videos])
        path = self.sanitized_path()
        with open(path, 'w') as f:
            f.write("ffconcat version 1.0\n# Written %s\n%s\n" % (
                datetime.datetime.now(), video_list))
            f.write("file '%s'\n" %
                    os.path.basename(queue_after.path))

    def has_video(self, video):
        '''
        Return true if the queue has any videos
        '''
        return video in self.videos

    def sanitized_path(self, archive=False, unfinished=False):
        '''
        Return sanitized path for any queue filename/path, etc.
        '''
        filename = os.path.basename(self.path)
        if unfinished:
            name, ext = os.path.splitext(filename)
            filename = ''.join([name, '.unfinished', ext])
        if archive or self.is_archived:
            return os.path.join(QUEUE_PATH, 'old_lists', filename)
        return os.path.join(QUEUE_PATH, filename)

    def duration(self):
        return reduce(
            operator.add,
            [v.duration for v in self.videos],
            datetime.timedelta(hours=0)
        )

    def archive(self):
        '''
        Archive provided queue
        '''
        archive_path = self.sanitized_path(archive=True)
        os.rename(self.sanitized_path(), archive_path)
        s3.upload_file(archive_path, 'archived_file_lists/' + self.path)
