import boto3
import botocore
import os
import socket
from settings import SETTINGS


def put(key, value, include_hostname=True):
    '''
    Writes to a given key in s3 as text. Readable by get
    '''
    if include_hostname:
        key = socket.gethostname() + "/" + key
    return _bucket().put_object(Key=key, Body=value)


def get(key, include_hostname=True):
    '''
    Reads from a given key in s3 as text. Writable by put
    '''
    try:
        if include_hostname:
            key = socket.gethostname() + "/" + key
        return _client().get_object(
            Bucket=SETTINGS['s3']['bucket_name'],
            Key=key
        )['Body'].read()
    except botocore.exceptions.ClientError as e:
        if "NoSuchKey" in str(e):
            return None
        raise


def list(prefix):
    '''
    Lists all files starting with the prefix.
    '''
    return _bucket().objects.filter(Prefix=prefix)


def upload_file(file_path, key, include_hostname=True):
    '''
    Uploads a file to s3 at a given key. If the hostname is not included,
    places the file at root.
    '''
    if include_hostname:
        key = socket.gethostname() + "/" + key
    return _bucket().upload_file(os.path.join(file_path), key)


def download_file(file_path, key):
    '''
    Downloads a file to s3 at a given key. If the hostname is not included,
    downloads the file from root.
    '''
    return _bucket().download_file(os.path.join(file_path), key)


def _bucket():
    '''
    Env configured boto3 bucket
    '''
    return boto3.resource(
        's3',
        aws_access_key_id=SETTINGS['s3']['aws_access_key_id'],
        aws_secret_access_key=SETTINGS['s3']['aws_secret_access_key']
    ).Bucket(SETTINGS['s3']['bucket_name'])


def _client():
    '''
    Env configured boto3 client
    '''
    return boto3.Session(
        aws_access_key_id=SETTINGS['s3']['aws_access_key_id'],
        aws_secret_access_key=SETTINGS['s3']['aws_secret_access_key']
    ).client('s3')
